/*
    Copyright (C) 2017 Red Hat, Inc.

    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at

            http://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
*/
import { ExpressionModel } from './expression.model';
import { FieldAction } from './field-action.model';

export enum TransitionMode {
  ONE_TO_ONE = 'ONE_TO_ONE',
  ONE_TO_MANY = 'ONE_TO_MANY',
  ENUM = 'ENUM',
  MANY_TO_ONE = 'MANY_TO_ONE',
  FOR_EACH = 'FOR_EACH',
  EXPRESSION = 'EXPRESSION',
}

export enum TransitionDelimiter {
  NONE,
  AMPERSAND,
  AT_SIGN,
  BACKSLASH,
  COLON,
  COMMA,
  DASH,
  EQUAL,
  HASH,
  PERIOD,
  PIPE,
  SEMICOLON,
  SLASH,
  SPACE,
  UNDERSCORE,
  USER_DEFINED,
}

export class TransitionDelimiterModel {
  delimiter: TransitionDelimiter = TransitionDelimiter.SPACE;
  serializedValue: string | null = null;
  prettyName: string | null = null;
  actualDelimiter = '';

  constructor(
    delimiter: TransitionDelimiter,
    serializedValue: string | null,
    prettyName: string,
    actualDelimiter: string
  ) {
    this.delimiter = delimiter;
    this.serializedValue = serializedValue;
    this.prettyName = prettyName;
    this.actualDelimiter = actualDelimiter;
  }
}

export class TransitionModel {
  static delimiterModels: TransitionDelimiterModel[] = [];

  mode: TransitionMode = TransitionMode.ONE_TO_ONE;
  delimiter: TransitionDelimiter = TransitionDelimiter.SPACE;
  userDelimiter = '';
  lookupTableName: string | null = null;
  expression!: ExpressionModel;
  enableExpression = false;
  transitionFieldAction: FieldAction | null = null;

  constructor() {
    TransitionModel.initialize();
  }

  static initialize() {
    if (TransitionModel.delimiterModels.length === 0) {
      const models: TransitionDelimiterModel[] = [];
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.NONE,
          null,
          '[None]',
          ''
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.AMPERSAND,
          'Ampersand',
          'Ampersand [&]',
          '&'
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.AT_SIGN,
          'AtSign',
          'At Sign [@]',
          '@'
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.BACKSLASH,
          'Backslash',
          'Backslash [\\]',
          '\\'
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.COLON,
          'Colon',
          'Colon [:]',
          ':'
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.COMMA,
          'Comma',
          'Comma [,]',
          ','
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.DASH,
          'Dash',
          'Dash [-]',
          '-'
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.EQUAL,
          'Equal',
          'Equal [=]',
          '='
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.HASH,
          'Hash',
          'Hash [#]',
          '#'
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.PERIOD,
          'Period',
          'Period [.]',
          '.'
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.PIPE,
          'Pipe',
          'Pipe [|]',
          '|'
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.SEMICOLON,
          'Semicolon',
          'Semicolon [;]',
          ';'
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.SLASH,
          'Slash',
          'Slash [/]',
          '/'
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.SPACE,
          'Space',
          'Space [ ]',
          ' '
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.UNDERSCORE,
          'Underscore',
          'Underscore [_]',
          '_'
        )
      );
      models.push(
        new TransitionDelimiterModel(
          TransitionDelimiter.USER_DEFINED,
          'User defined',
          'User defined',
          '\0'
        )
      );
      TransitionModel.delimiterModels = models;
    }
  }

  static delimiterToModel(delimiter: string): TransitionDelimiterModel {
    return TransitionModel.delimiterModels.filter(
      (model) => model.actualDelimiter === delimiter
    )[0];
  }

  /**
   * Translate a mapping mode number into a string.
   * @param mode
   */
  static getMappingModeName(mode: TransitionMode): string {
    let actionName: string;

    switch (mode) {
      case TransitionMode.ONE_TO_ONE: {
        actionName = 'One to One';
        break;
      }
      case TransitionMode.MANY_TO_ONE: {
        actionName = 'Many to One';
        break;
      }
      case TransitionMode.ONE_TO_MANY: {
        actionName = 'One to Many';
        break;
      }
      case TransitionMode.ENUM: {
        actionName = 'ENUM';
        break;
      }
      case TransitionMode.FOR_EACH: {
        actionName = 'For Each';
        break;
      }
      case TransitionMode.EXPRESSION: {
        actionName = 'Expression';
        break;
      }
      default: {
        actionName = '';
        break;
      }
    }
    return actionName;
  }

  getPrettyName() {
    let transitionFieldActionName = '';
    if (this.transitionFieldAction) {
      transitionFieldActionName = ' (' + this.transitionFieldAction.name + ')';
    }
    if (
      this.mode === TransitionMode.ONE_TO_MANY ||
      this.mode === TransitionMode.MANY_TO_ONE
    ) {
      return (
        TransitionModel.getMappingModeName(this.mode) +
        transitionFieldActionName
      );
    } else if (this.mode === TransitionMode.ENUM) {
      return 'Enum (table: ' + this.lookupTableName + ')';
    } else if (this.mode === TransitionMode.EXPRESSION) {
      return 'Expression (' + this.expression?.toText() + ')';
    }
    return TransitionModel.getMappingModeName(this.mode);
  }

  isOneToManyMode(): boolean {
    return this.mode === TransitionMode.ONE_TO_MANY;
  }

  isOneToOneMode(): boolean {
    return this.mode === TransitionMode.ONE_TO_ONE;
  }

  isManyToOneMode(): boolean {
    return this.mode === TransitionMode.MANY_TO_ONE;
  }

  isForEachMode(): boolean {
    return this.mode === TransitionMode.FOR_EACH;
  }

  isEnumerationMode(): boolean {
    return this.mode === TransitionMode.ENUM;
  }

  isExpressionMode(): boolean {
    return this.mode === TransitionMode.EXPRESSION;
  }
}
