/*
 * #%L
 * Wildfly Camel :: Testsuite
 * %%
 * Copyright (C) 2013 - 2015 RedHat
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

package org.wildfly.camel.test.infinispan;

import java.util.HashMap;
import java.util.Map;

import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.infinispan.InfinispanConstants;
import org.infinispan.manager.CacheContainer;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.wildfly.extension.camel.CamelAware;
import org.wildfly.extension.camel.WildFlyCamelContext;

@CamelAware
@RunWith(Arquillian.class)
public class InfinispanIntegrationTest {

    private static final String CONTAINER_NAME = "java:jboss/infinispan/container/server";

    private static final String CACHE_KEY_NAME = "name";
    private static final String CACHE_KEY_AGE = "age";
    private static final String CACHE_VALUE_KERMIT = "Kermit";
    private static final String CACHE_VALUE_BOB = "Bob";
    private static final int CACHE_VALUE_AGE = 65;

    private CacheContainer cacheContainer;
    private WildFlyCamelContext camelctx;
    private String cacheName;

    @Deployment
    public static WebArchive deployment() {
        WebArchive archive = ShrinkWrap.create(WebArchive.class, "camel-infinispan-test.war");
        archive.addAsWebInfResource("infinispan/web.xml", "web.xml");
        return archive;
    }

    @Before
    public void setUp() throws Exception {
        camelctx = new WildFlyCamelContext();
        cacheContainer = (CacheContainer) camelctx.getNamingContext().lookup(CONTAINER_NAME);
        cacheName = cacheContainer.getCache().getName();
    }

    @Test
    public void testCacheGet() throws Exception {

        camelctx.addRoutes(new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                from("direct:get")
                .to("infinispan://" + cacheName + "?cacheContainer=#" + CONTAINER_NAME + "&command=GET")
                .to("mock:end");
            }
        });

        cacheContainer.getCache().put(CACHE_KEY_NAME, CACHE_VALUE_KERMIT);

        camelctx.start();
        try {
            Map<String, Object> headers = new HashMap<>();
            headers.put(InfinispanConstants.KEY, CACHE_KEY_NAME);

            ProducerTemplate template = camelctx.createProducerTemplate();
            String name = template.requestBodyAndHeaders("direct:get", null, headers, String.class);

            Assert.assertEquals("Kermit", name);
        } finally {
            camelctx.stop();
        }
    }

    @Test
    public void testCachePut() throws Exception {

        camelctx.addRoutes(new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                from("direct:put")
                .to("infinispan://" + cacheName + "?cacheContainer=#" + CONTAINER_NAME + "&command=PUT");
            }
        });

        camelctx.start();
        try {
            Map<String, Object> headers = new HashMap<>();
            headers.put(InfinispanConstants.KEY, CACHE_KEY_NAME);
            headers.put(InfinispanConstants.VALUE, CACHE_VALUE_KERMIT);

            ProducerTemplate template = camelctx.createProducerTemplate();
            template.requestBodyAndHeaders("direct:put", null, headers);

            String name = (String) cacheContainer.getCache().get(CACHE_KEY_NAME);
            Assert.assertEquals("Kermit", name);
        } finally {
            camelctx.stop();
        }
    }

    @Test
    public void testCacheRemove() throws Exception {

        camelctx.addRoutes(new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                from("direct:remove")
                .to("infinispan://" + cacheName + "?cacheContainer=#" + CONTAINER_NAME + "&command=REMOVE");
            }
        });

        cacheContainer.getCache().put(CACHE_KEY_NAME, CACHE_VALUE_KERMIT);

        camelctx.start();
        try {
            Assert.assertTrue(cacheContainer.getCache().containsKey(CACHE_KEY_NAME));

            Map<String, Object> headers = new HashMap<>();
            headers.put(InfinispanConstants.KEY, CACHE_KEY_NAME);

            ProducerTemplate template = camelctx.createProducerTemplate();
            template.requestBodyAndHeaders("direct:remove", null, headers);

            Assert.assertFalse(cacheContainer.getCache().containsKey(CACHE_KEY_NAME));
        } finally {
            camelctx.stop();
        }
    }
}
