package io.apicurio.registry.systemtests;

import io.apicur.registry.v1.ApicurioRegistry;
import io.fabric8.junit.jupiter.api.KubernetesTest;
import io.fabric8.junit.jupiter.api.LoadKubernetesManifests;
import io.fabric8.openshift.client.OpenShiftClient;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import static io.apicurio.registry.systemtests.Utils.findOperatorDeployment;
import static io.apicurio.registry.systemtests.Utils.isDeploymentReady;
import static io.apicurio.registry.systemtests.Utils.waitDeploymentReady;
import static org.junit.jupiter.api.Assertions.assertTrue;

@KubernetesTest
@LoadKubernetesManifests({
        // Shared resources
        "/common/00_operator_group.yaml", // Operator group for installed operators in namespace
        // PostgreSQL database resources
        "/sql/00_service.yaml", // Service
        "/sql/01_deployment.yaml", // Deployment
        // Apicurio Registry operator
        "/apicurio/01_subscription.yaml", // Apicurio Registry operator subscription
        // Apicurio Registry instance
        "/apicurio/02_registry_sql_no_iam.yaml" // Apicurio Registry instance with PostgreSQL storage and without IAM
})
public class SqlNoIAM extends TestBase {
    /** {@link OpenShiftClient} instance for tests. */
    private OpenShiftClient client;

    /**
     * Performs actions that should be done before each test.
     */
    @BeforeEach
    public void beforeEach() {
        // Log information about current action
        logger.info("Running BeforeEach actions.");

        // Wait for readiness of PostgreSQL database deployment
        assertTrue(waitDeploymentReady(client, Constants.POSTGRESQL_NAME));

        // Wait for readiness of Apicurio Registry operator deployment
        assertTrue(waitDeploymentReady(
                client,
                findOperatorDeployment(client, Constants.REGISTRY_OPERATOR_NAME).getMetadata().getName())
        );

        // Log information about current action
        logger.info("BeforeEach finished.");
    }

    /**
     * Tests that {@link ApicurioRegistry} with PostgreSQL database storage becomes ready.
     */
    @Test
    public void testDeploy() {
        // Log information about current action
        logger.info("Running testDeploy test.");

        // Wait for deployment readiness of Apicurio Registry instance with PostgreSQL database storage and without IAM
        assertTrue(waitDeploymentReady(client, Constants.REGISTRY_NAME + "-deployment"));

        // Check deployment readiness of Apicurio Registry instance with PostgreSQL database storage and without IAM
        assertTrue(isDeploymentReady(client, Constants.REGISTRY_NAME + "-deployment"));

        // Log information about current action
        logger.info("testDeploy test finished.");
    }
}
