package org.jboss.windup.tooling;

import java.rmi.Remote;
import java.rmi.RemoteException;
import java.util.logging.LogRecord;

/**
 * Extends the standard progress monitor to also tie in with the logging system. Messages from Java Util Logging (aka JUL) will be passed through this
 * if the provided listener is an instance of {@link WindupToolingProgressMonitor}.
 *
 * @author <a href="mailto:jesse.sightler@gmail.com">Jesse Sightler</a>
 */
public interface WindupToolingProgressMonitor extends Remote {
    /**
     * Constant indicating an unknown amount of work.
     */
    int UNKNOWN = -1;

    /**
     * Called for each {@link LogRecord} generated by Windup.
     */
    void logMessage(LogRecord logRecord) throws RemoteException;

    /**
     * Notifies that the main task is beginning. This must only be called once on a given progress monitor instance.
     *
     * @param name      the name (or description) of the main task
     * @param totalWork the total number of work units into which the main task is been subdivided. If the value is <code>UNKNOWN</code> the
     *                  implementation is free to indicate progress in a way which doesn't require the total number of work units in advance.
     */
    void beginTask(String name, int totalWork) throws RemoteException;

    /**
     * Notifies that the work is done; that is, either the main task is completed or the user canceled it. This method may be called more than once
     * (implementations should be prepared to handle this case).
     */
    void done() throws RemoteException;

    /**
     * Returns whether cancellation of current operation has been requested. Long-running operations should poll to see if cancelation has been
     * requested.
     *
     * @return <code>true</code> if cancellation has been requested, and <code>false</code> otherwise
     * @see #setCancelled(boolean)
     */
    boolean isCancelled() throws RemoteException;

    /**
     * Sets the cancel state to the given value.
     *
     * @param value <code>true</code> indicates that cancellation has been requested (but not necessarily acknowledged); <code>false</code> clears
     *              this flag
     * @see #isCancelled()
     */
    void setCancelled(boolean value) throws RemoteException;

    /**
     * Sets the task name to the given value. This method is used to restore the task label after a nested operation was executed. Normally there is
     * no need for clients to call this method.
     *
     * @param name the name (or description) of the main task
     * @see #beginTask(java.lang.String, int)
     */
    void setTaskName(String name) throws RemoteException;

    /**
     * Notifies that a subtask of the main task is beginning. Subtasks are optional; the main task might not have subtasks.
     *
     * @param name the name (or description) of the subtask
     */
    void subTask(String name) throws RemoteException;

    /**
     * Notifies that a given number of work unit of the main task has been completed. Note that this amount represents an installment, as opposed to a
     * cumulative amount of work done to date.
     *
     * @param work a non-negative number of work units just completed
     */
    void worked(int work) throws RemoteException;

}
