/*******************************************************************************
 * Copyright (c) 2018 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.kernel.service.util;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import org.eclipse.osgi.report.resolution.ResolutionReport;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.framework.hooks.resolver.ResolverHook;
import org.osgi.framework.hooks.resolver.ResolverHookFactory;
import org.osgi.framework.wiring.BundleCapability;
import org.osgi.framework.wiring.BundleRequirement;
import org.osgi.framework.wiring.BundleRevision;
import org.osgi.resource.Resource;

import com.ibm.websphere.ras.Tr;
import com.ibm.websphere.ras.TraceComponent;

/**
 * Utility to assist in getting a resolution report generated by the equinox framework during a
 * bundle resolution operation.
 *
 * This utility class is not thread safe. It should be held on thread stack by the same thread
 * triggering the resolve operation. Calls should be made to startHelper() and stopHelper()
 * right before and after the resolve. The call to stopHelper() should be made in a finally to
 * ensure the resolver hook is unregistered. For example:
 *
 * <pre>
{@code}
            ...
            ResolutionReportHelper rrh = null;
            if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled()) {
                rrh = new ResolutionReportHelper();
                rrh.startHelper(bContext);
            }
            try {
               // any operation which triggers a bundle resolve
                wiring.resolveBundles(bundlesToResolve);
            } finally {
                if (rrh != null) {
                    rrh.stopHelper(bContext);
                    // output the resolution report at desired trace level.
                    Tr.debug(this, tc, rrh.getReportSumamry());
                }
            }
 * </pre>
 */
public class ResolutionReportHelper implements ResolverHookFactory, ResolverHook, ResolutionReport.Listener {
    private static final TraceComponent tc = Tr.register(ResolutionReportHelper.class);

    private final AtomicReference<ResolutionReport> _reportRef = new AtomicReference<ResolutionReport>();
    private ServiceRegistration<ResolverHookFactory> ResolverHookFactoryReg;

    private Long _resolvingThread = null;

    /**** overrides from ResolverHookFactory *****/
    @Override
    public ResolverHook begin(Collection<BundleRevision> triggers) {
        ResolverHook hook = null;
        // Only return hook if resolve is on behalf of the thread that started this handler;
        if (_resolvingThread == Thread.currentThread().getId()) {
            hook = this;
        }
        return hook;
    }

    /**** overrides from ResolutionReport.Listener ****/
    @Override
    public void handleResolutionReport(ResolutionReport report) {
        // If there are multiple calls, just capture first one.
        _reportRef.compareAndSet(null, report);
    }

    /**** overrides from ResolverHook ****/
    @Override
    public void filterResolvable(Collection<BundleRevision> candidates) {}

    @Override
    public void filterSingletonCollisions(BundleCapability singleton,
                                          Collection<BundleCapability> collisionCandidates) {}

    @Override
    public void filterMatches(BundleRequirement requirement, Collection<BundleCapability> candidates) {}

    @Override
    public void end() {}

    /** Utility methods for code that instantiated this object */

    /**
     * Should be called just prior to the resolve bundle operation from the same thread.
     *
     * @param bContext bundleContext
     */
    public void startHelper(BundleContext bContext) {
        _resolvingThread = Thread.currentThread().getId();
        ResolverHookFactoryReg = bContext.registerService(ResolverHookFactory.class, this, null);
    }

    /**
     * Invoke from the same thread as the call to startHelper, as soon as the resolve completes. Invoke
     * from a finally block to guarantee it's called.
     */
    public void stopHelper() {
        _resolvingThread = null;
        if (ResolverHookFactoryReg != null)
            ResolverHookFactoryReg.unregister();
    }

    public ResolutionReport getResolutionReport() {
        return _reportRef.get();
    }

    /**
     * Convenience method to get a string to use for debug tracing.
     *
     * @return A string describing any unresolved resources or resolution exception.
     */
    public String getResolutionReportString() {
        ResolutionReport report = _reportRef.get();
        StringBuilder reportMessage = null;
        if (report != null) {
            Map<Resource, List<ResolutionReport.Entry>> unresolved;
            reportMessage = new StringBuilder().append("Resolver report:\n");
            unresolved = report.getEntries();
            if (!unresolved.isEmpty()) {
                for (Resource resource : unresolved.keySet()) {
                    reportMessage.append(report.getResolutionReportMessage(resource));
                }
            }
            if (report.getResolutionException() != null) {
                reportMessage.append("\nResolution exception:\n" + report.getResolutionException());
            }
        }
        return reportMessage == null ? "No resolution report generated." : reportMessage.toString();
    }
}
