/*******************************************************************************
 * Copyright (c) 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package javax.servlet.sip;


/**
 * Factory interface for a variety of SIP Servlet API abstractions.
 * 
 * <p>SIP servlet containers are requried to make a
 * <code>SipFactory</code> instance available to applications
 * through a <code>ServletContext</code> attribute with name
 * <code>javax.servlet.sip.SipFactory</code>.
 * 
 */
public interface SipFactory {
    /**
     * Returns a <code>Address</code> corresponding to the specified
     * string. The resulting object can be used, for example, as the
     * value of From or To headers of locally initiated SIP requests.
     * 
     * <p>The special argument "*" results in a <em>wildcard</em>
     * <code>Address</code> being returned, that is, an <code>Address</code>
     * for which <code>isWildcard</code> returns true. Such addresses are
     * for use in <code>Contact</code> headers only.
     * 
     * @param sipAddress valid value of SIP From or To header
     * @return a parsed <code>Address</code>
     * @throws ServletParseException if parsing failed
     */
    Address createAddress(String sipAddress) throws ServletParseException;

    /**
     * Returns an <code>Address</code> with the specified URI and no
     * display name.
     * 
     * @param uri   the <code>URI</code> of the returned <code>Address</code>
     * @return  an <code>Address</code> whose <code>URI</code>
     *          component is the argument
     */
    Address createAddress(URI uri);
    
    /**
     * Returns a new <code>Address</code> with the specified
     * <code>URI</code> and display name.
     * 
     * @param uri           <code>URI</code> of the new <code>Address</code>
     * @param displayName   display name of the new <code>Address</code>
     * 
     * @return created address
     */
    Address createAddress(URI uri, String displayName);
    
    
    /**
     * Returns a new <code>SipApplicationSession</code>. This is useful,
     * for example, when an application is being initialized and wishes
     * to perform some signaling action.
     * 
     * @return a new <code>SipApplicationSession</code> object
     */
    SipApplicationSession createApplicationSession();
    
    /**
     * 
     * Returns a new SipApplicationSession identified by the specified 
     * SipApplicationKey. This is same as the one generated by the method 
     * annotated with @SipApplicationKey annotation. This allows a way to 
     * associate incoming requests to an already existing SipApplicationSession. 
     * 
     * @param sipApplicationKey - id for the SipApplicationSession
     * 
     * @return a new SipApplicationSession object with the specified id
     */
    SipApplicationSession createApplicationSessionByKey(String sipApplicationKey);
    
    
    /**
     * Creates a new AuthInfo object that can be used to provide authentication 
     * information on servlet initiated requests.
     *  
     * @return AuthInfo a new instance of AuthInfo
     */
    AuthInfo createAuthInfo();
    
    /**
     * 
     * Creates a new Parameterable parsed from the specified string. 
     * The string must be in the following format:
     * 
     * field-value *(;parameter-name[=parameter-value])
     *  
     * where the field-value may be in name-addr or addr-spec format 
     * as defined in RFC 3261 or may be any sequence of tokens till the 
     * first semicolon. 
     * 
     * @param paramString - the header field string 
     * @return a parsed Parameterable
     * @throws ServletParseException
     */
    Parameterable createParameterable(String paramString) throws ServletParseException;
    
    /**
     * Returns a new request object with the specified request method, From,
     * and To headers. The returned request object exists in a new
     * <code>SipSession</code> which belongs to the specified
     * <code>SipApplicationSession</code>.
     * 
     * <p>This method is used by servlets acting as SIP clients in order to
     * send a request in a new call leg. The container is responsible for 
     * assigning the request appropriate Call-ID and CSeq headers, as well as
     * Contact header <em>if</em> the method is not REGISTER.
     * 
     * <p>This method makes a copy of the from and to arguments and
     * associates them with the new <code>SipSession</code>. Any component
     * of the from and to URIs not allowed in the context of SIP
     * From and To headers are removed from the copies. This includes,
     * headers and various parameters. Also, a "tag" parameter in either
     * of the copied from or to is also removed, as it is illegal in an
     * initial To header and the container will choose it's own tag for
     * the From header. The copied from and to addresses can be obtained
     * from the SipSession but must not be modified by applications.
     *
     * @param appSession    the application session to which the new
     *          <code>SipSession</code> and <code>SipServletRequest</code>
     *          belongs
     * @param method    the method of the new request, e.g. "INVITE"
     * @param from      value of the From header
     * @param to        value of the To header
     * @return  the request object with method, request URI,
     *          and From, To, Call-ID, CSeq, Route headers filled in.
     * @throws IllegalArgumentException - if the method is "ACK" or "CANCEL", or the 
     * 							specified SipApplicationSession is invalid.
     */
    SipServletRequest createRequest(
            SipApplicationSession appSession,
            String method,
            Address from,
            Address to) throws IllegalArgumentException;
    
    /**
     * Returns a new request object with the specified request method, From,
     * and To headers. The returned request object exists in a new
     * <code>SipSession</code> which belongs to the specified
     * <code>SipApplicationSession</code>.
     * 
     * <p>This method is used by servlets acting as SIP clients in order to
     * send a request in a new call leg. The container is responsible for 
     * assigning the request appropriate Call-ID and CSeq headers, as well as
     * Contact header <em>if</em> the method is not REGISTER.
     * 
     * <p>This method is functionally equivalent to:
     * <pre>
     * createRequest(method, f.createAddress(from), f.createAddress(to));
     * </pre>
     * Note that this implies that if either of the from or to argument
     * is a SIP URI containing parameters, the URI must be enclosed in
     * angle brackets. Otherwise the address will be parsed as if the
     * parameter belongs to the address and not the URI.
     *
     * @param appSession    the application session to which the new
     *          <code>SipSession</code> and <code>SipServletRequest</code>
     *          belongs
     * @param method the method of the new request, e.g. "INVITE"
     * @param from  value of the From header -- this must be a
     *              valid <code>Address</code>
     * @param to    value of the To header -- this must be a
     *              valid <code>Address</code>
     * @return the request object with method, request URI,
     *         and From, To, Call-ID, CSeq, Route headers filled in.
     * @throws ServletParseException if the URI scheme of
               the from or to argument is unknown or if parsing failed
     * @throws IllegalArgumentException - if the method is "ACK" or "CANCEL", or the 
     * 		   specified SipApplicationSession is invalid. 
     */
    SipServletRequest createRequest(
            SipApplicationSession appSession,
            String method,
            String from,
            String to) throws IllegalArgumentException, ServletParseException;
    
    /**
     * Returns a new request object with the specified request method, From,
     * and To headers. The returned request object exists in a new
     * <code>SipSession</code> which belongs to the specified
     * <code>SipApplicationSession</code>.
     * 
     * <p>This method is used by servlets acting as SIP clients in order to
     * send a request in a new call leg. The container is responsible for 
     * assigning the request appropriate Call-ID and CSeq headers, as well as
     * Contact header <em>if</em> the method is not REGISTER.
     * 
     * <p>This method makes a copy of the from and to arguments and
     * associates them with the new <code>SipSession</code>. Any component
     * of the from and to URIs not allowed in the context of SIP
     * From and To headers are removed from the copies. This includes,
     * headers and various parameters.
     * The from and to addresses can subsequently be obtained from the
     * SipSession or the returned request object but must not be modified
     * by applications.
     *
     * @param appSession    the application session to which the new
     *          <code>SipSession</code> and <code>SipServletRequest</code>
     *          belongs
     * @param method    the method of the new request, e.g. "INVITE"
     * @param from      value of the From header
     * @param to        value of the To header
     * @return  the request object with method, request URI,
     *          and From, To, Call-ID, CSeq, Route headers filled in.
     * @throws IllegalArgumentException if the method is "ACK" or "CANCEL"
     */
    SipServletRequest createRequest(
            SipApplicationSession appSession,
            String method,
            URI from,
            URI to) throws IllegalArgumentException;
    
    /**
     * 
     * @deprecated. usage of this method is deprecated. Setting the sameCallId 
     * flag to "true" actually breaks the provisions of [RFC 3261] where the Call-ID 
     * value is to be unique accross dialogs. Instead use a more general method defined 
     * on the B2buaHelper B2buaHelper.createRequest(SipServletRequest)
     * 
     * Creates a new request object belonging to a new SipSession. The
     * new request is similar to the specified <code>origRequest</code>
     * in that the method and the majority of header fields are copied
     * from <code>origRequest</code> to the new request.
     *
     * <p>This method satisfies the following rules:
     * <ul>
     * <li>The From header field of the new request has a new tag chosen
     *     by the container.
     * <li>The To header field of the new request has no tag.
     * <li>If the <code>sameCallId</code> argument is false, the new
     *     request (and the corresponding <code>SipSession</code>)is
     *     assigned a new Call-ID.
     * <li>Record-Route and Via header fields are not copied. As usual,
     *     the container will add its own Via header field to the request
     *     when it's actually sent outside the application server.
     * <li>For non-REGISTER requests, the Contact header field is not
     *     copied but is populated by the container as usual.
     * </ul>
     * 
     * <p>This method provides a convenient and efficient way of
     * constructing the second "leg" of a B2BUA application. It is used
     * only for the initial request. Subsequent requests in either leg
     * must be created using {@link SipSession#createRequest} as usual.
     *
     * @param origRequest request to be "copied"
     * @param sameCallId whether or not to use same Call-ID for the new
     *     dialog
     * @return  the "copied" request object
     */
    SipServletRequest createRequest(SipServletRequest origRequest,
                                    boolean sameCallId);

    /**
     * Constructs a <code>SipURI</code> with the specified user and host
     * components. The scheme will initially be <code>sip</code> but the
     * application may change it to <code>sips</code> by calling
     * <code>setSecure(true)</code> on the returned <code>SipURI</code>.
     * Likewise, the port number of the new URI is left unspecified but
     * may subsequently be set by calling <code>setPort</code> on the
     * returned <code>SipURI</code>.
     * 
     * @param user user part of the new <code>SipURI</code>
     * @param host host part of the new <code>SipURI</code>
     * 
     * @return new <em>insecure</em> <code>SipURI</code> with the specified
     *     user and host parts
     */
    SipURI createSipURI(String user, String host);

    /**
     * Returns a <code>URI</code> object corresponding to the specified
     * string. The <code>URI</code> may then be used as request URI in
     * SIP requests or as the URI component of {@link Address} objects.
     * 
     * <p>Implementations must be able to represent URIs of any scheme.
     * This method returns a <code>SipURI</code> object if the specified
     * string is a <code>sip</code> or a <code>sips</code> URI, and a
     * <code>TelURL</code> object if it's a <code>tel</code> URL.
     * 
     * @param uri   the URI string to parse
     * 
     * @return      a parsed <code>URI</code> object
     * 
     * @throws ServletParseException if the URI scheme
     *          is unknown or parsing failed
     */
    URI createURI(String uri) throws ServletParseException;
   
}
