/*******************************************************************************
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.adaptable.module.internal;

import java.util.HashMap;
import java.util.Map;

import com.ibm.wsspi.adaptable.module.Container;
import com.ibm.wsspi.adaptable.module.DefaultNotification;
import com.ibm.wsspi.adaptable.module.Notifier;
import com.ibm.wsspi.artifact.ArtifactContainer;
import com.ibm.wsspi.artifact.ArtifactNotifier;
import com.ibm.wsspi.artifact.ArtifactNotifier.ArtifactListener;
import com.ibm.wsspi.artifact.ArtifactNotifier.ArtifactNotification;
import com.ibm.wsspi.artifact.DefaultArtifactNotification;

/**
 * This is the default implementation of the {@link Notifier} interface.
 */
public class NotifierImpl implements Notifier {

    /** The root Container this notifier is for, will be used in all notification events generated by the notifier */
    private final Container root;
    /** The Artifact API notifier that this notifier delegates to */
    private final ArtifactNotifier delegateNotifier;
    /** The root artifact API container that this notifier is registered for */
    private final ArtifactContainer delegateRoot;
    /** A map of all of the listeners registered against this notifier to the DelegateListener that is registered against the {@link #delegateNotifier} */
    private final Map<NotificationListener, DelegateListener> listenerDelegates = new HashMap<NotificationListener, DelegateListener>();

    /**
     * @param delegateRoot The root ArtifactContainer this notifier can delegate to
     * @param root The root Container this notifier is for, will be used in all notification events generated by the notifier
     */
    public NotifierImpl(ArtifactContainer delegateRoot, Container root) {
        this.delegateRoot = delegateRoot;
        this.delegateNotifier = delegateRoot.getArtifactNotifier();
        this.root = root;
    }

    /**
     * This method checks that the root of the container on the target matches the one that we are the notifier for.
     *
     * @param targets The notification object being targeted
     * @throws IllegalArgumentException if the root of the container being targeted doesn't match the root for this notifier
     */
    private void verifyTargets(Notification targets) {
        if (targets.getContainer().getRoot() != this.root) {
            throw new IllegalArgumentException();
        }
    }

    /** {@inheritDoc} */
    @Override
    public boolean registerForNotifications(Notification targets, NotificationListener callbackObject) throws IllegalArgumentException {
        verifyTargets(targets);

        /*
         * Convert the target artifact version so we can register with the delegate. We just use the delegate root (as we can't convert the Container supplied on the target to an
         * ArtifactContainer), we know this is correct because we ran it through verifyTargets first.
         */
        ArtifactNotification artifactNotification = new DefaultArtifactNotification(this.delegateRoot, targets.getPaths());

        // Create a listener that will listen to the artifact notifier then notify the callback for this registration when things change
        DelegateListener listener = new DelegateListener(this.root, callbackObject);
        boolean isRegistered = this.delegateNotifier.registerForNotifications(artifactNotification, listener);
        if (isRegistered) {
            // We could register with the delegate so keep track of the listener (in case we need to unregister it later)
            this.listenerDelegates.put(callbackObject, listener);
        }
        return isRegistered;
    }

    /** {@inheritDoc} */
    @Override
    public boolean removeListener(NotificationListener listenerToRemove) {
        // Get the listener that is registered with the delegate and remove it
        ArtifactListener delegateListener = this.listenerDelegates.remove(listenerToRemove);
        return this.delegateNotifier.removeListener(delegateListener);
    }

    /** {@inheritDoc} */
    @Override
    public boolean setNotificationOptions(long interval, boolean useMBean) {
        return this.delegateNotifier.setNotificationOptions(interval, useMBean);
    }

    /**
     * This is an {@link ArtifactListener} that will convert any {@link ArtifactListener#notifyEntryChange(ArtifactNotification, ArtifactNotification, ArtifactNotification)} call
     * into a {@link NotificationListener#notifyEntryChange(Notification, Notification, Notification)} call.
     */
    private class DelegateListener implements com.ibm.ws.artifact.ArtifactNotifierExtension.ArtifactListener {

        private final NotificationListener listenerToInform;
        private String id;

        /**
         * @param root
         * @param listenerToInform
         */
        public DelegateListener(Container root, NotificationListener listenerToInform) {
            super();
            this.listenerToInform = listenerToInform;

            // If the listener has an Id, save it.
            if (listenerToInform instanceof com.ibm.ws.adaptable.module.NotifierExtension.NotificationListener) {
                id = ((com.ibm.ws.adaptable.module.NotifierExtension.NotificationListener) listenerToInform).getId();
            }
        }

        @Override
        public void notifyEntryChange(ArtifactNotification added, ArtifactNotification removed, ArtifactNotification modified) {
            notifyEntryChange(added, removed, modified, null);
        }

        @Override
        public void notifyEntryChange(ArtifactNotification added, ArtifactNotification removed, ArtifactNotification modified, String filter) {
            /*
             * Our delegate called us so we need to call all of our listeners having converted the artifact notification to notifications objects. We know these must be under the
             * root for this notifier so just use that in all the new notificaitons
             */
            Notification addedNotification = new DefaultNotification(root, added.getPaths());
            Notification removedNotification = new DefaultNotification(root, removed.getPaths());
            Notification modifiedNotification = new DefaultNotification(root, modified.getPaths());
            this.listenerToInform.notifyEntryChange(addedNotification, removedNotification, modifiedNotification);
        }

        @Override
        public String getId() {
            return id;
        }
    }

}
