/*******************************************************************************
 * Copyright (c) 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.crypto.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.Arrays;

import org.junit.Test;

/**
 * Tests for the password utility class.
 */
public class HashedDataTest {

    byte expectDefault[] = { 0x01, 0x30, 0x00, 0x00, 0x00, 0x04, 0x73, 0x61, 0x6C, 0x74, 0x40, 0x00, 0x00, 0x00, 0x20, 0x0A, (byte) (0xFC & 0xFF), (byte) (0xB9 & 0xFF),
                            (byte) (0xE1 & 0xFF), (byte) (0xC2 & 0xFF), 0x1A, (byte) (0xE2 & 0xFF), (byte) (0xCA & 0xFF), 0x56, 0x53, (byte) (0xF1 & 0xFF), (byte) (0xFD & 0xFF),
                            0x73, (byte) (0xC5 & 0xFF), (byte) (0x88 & 0xFF), (byte) (0x8D & 0xFF), 0x32, 0x0A, (byte) (0xFE & 0xFF), (byte) (0x97 & 0xFF), (byte) (0x99 & 0xFF),
                            (byte) (0xDB & 0xFF), 0x11, (byte) (0xFA & 0xFF), 0x13, (byte) (0xB5 & 0xFF), 0x16, 0x31, 0x6B, (byte) (0xB3 & 0xFF), (byte) (0xFD & 0xFF), 0x07 };

    byte expectCustom[] = { 0x01, 0x20, 0x00, 0x00, 0x03, (byte) (0xDD & 0xFF), 0x50, 0x00, 0x00, 0x00, 0x40, 0x30, 0x00, 0x00, 0x00, 0x05, 0x73, 0x61, 0x6C, 0x74, 0x32, 0x40,
                           0x00, 0x00, 0x00, 0x08, (byte) (0x90 & 0xFF), (byte) (0xCC & 0xFF), 0x41, 0x0D, 0x6D, (byte) (0x93 & 0xFF), (byte) (0xD1 & 0xFF), 0x1A };

    /**
     * Test constructor
     */
    @Test
    public void testConstructorDefault() {
        try {
            char plain[] = "password".toCharArray();
            byte salt[] = "salt".getBytes("UTF-8"); //cannot be autogenerated in order to compare with pre-hashed data.
            String algorithm = "PBKDF2WithHmacSHA1";
            int iteration = 6384;
            int length = 256;
            HashedData hd = new HashedData(plain, algorithm, salt, iteration, length, null);
            assertTrue(Arrays.equals(hd.getPlain(), plain));
            assertTrue(Arrays.equals(hd.getSalt(), salt));
            assertEquals(algorithm, hd.getAlgorithm());
            assertEquals(iteration, hd.getIteration());
            assertEquals(length, hd.getOutputLength());
            try {
                assertTrue(Arrays.equals(hd.toBytes(), expectDefault));
            } catch (Exception e) {
                fail("Unexpected exception : " + e.getMessage());
            }
        } catch (Exception e) {
            fail("Unexpected exception : " + e.getMessage());
        }
    }

    /**
     * Test constructor
     */
    @Test
    public void testConstructorCustom() {
        try {
            char plain[] = "password2".toCharArray();
            byte salt[] = "salt2".getBytes("UTF-8"); //cannot be autogenerated in order to compare with pre-hashed data.
            String algorithm = "PBKDF2WithHmacSHA1";
            int iteration = 989;
            int length = 64;
            HashedData hd = new HashedData(plain, algorithm, salt, iteration, length, null);
            assertTrue(Arrays.equals(hd.getPlain(), plain));
            assertTrue(Arrays.equals(hd.getSalt(), salt));
            assertEquals(algorithm, hd.getAlgorithm());
            assertEquals(iteration, hd.getIteration());
            assertEquals(length, hd.getOutputLength());
            try {
                assertTrue(Arrays.equals(hd.toBytes(), expectCustom));
            } catch (Exception e) {
                fail("Unexpected exception : " + e.getMessage());
            }
        } catch (Exception e) {
            fail("Unexpected exception : " + e.getMessage());
        }
    }

    /**
     * Test constructor
     */
    @Test
    public void testConstructorDigestDefault() {
        try {
            byte salt[] = "salt".getBytes("UTF-8"); //cannot be autogenerated in order to compare with pre-hashed data.
            String algorithm = "PBKDF2WithHmacSHA1";
            int iteration = 6384;
            int length = 256;
            HashedData hd = new HashedData(expectDefault);
            assertTrue(Arrays.equals(hd.getSalt(), salt));
            assertEquals(algorithm, hd.getAlgorithm());
            assertEquals(iteration, hd.getIteration());
            assertEquals(length, hd.getOutputLength());
        } catch (Exception e) {
            fail("Unexpected exception : " + e.getMessage());
        }
    }

    /**
     * Test constructor
     */
    @Test
    public void testConstructorDigestCustom() {
        try {
            byte salt[] = "salt2".getBytes("UTF-8"); //cannot be autogenerated in order to compare with pre-hashed data.
            String algorithm = "PBKDF2WithHmacSHA1";
            int iteration = 989;
            int length = 64;
            HashedData hd = new HashedData(expectCustom);
            assertTrue(Arrays.equals(hd.getSalt(), salt));
            assertEquals(algorithm, hd.getAlgorithm());
            assertEquals(iteration, hd.getIteration());
            assertEquals(length, hd.getOutputLength());
        } catch (Exception e) {
            fail("Unexpected exception : " + e.getMessage());
        }
    }

    /**
     * Test constructor with invalid algorithm
     */
    @Test
    public void testConstructorInvalidAlgorithm() {
        try {
            char plain[] = "password2".toCharArray();
            byte salt[] = "salt2".getBytes("UTF-8"); //cannot be autogenerated in order to compare with pre-hashed data.
            String algorithm = "DoesnotExist";
            int iteration = 989;
            int length = 128;
            HashedData hd = new HashedData(plain, algorithm, salt, iteration, length, null);
            assertTrue(Arrays.equals(hd.getPlain(), plain));
            assertTrue(Arrays.equals(hd.getSalt(), salt));
            assertEquals(algorithm, hd.getAlgorithm());
            assertEquals(iteration, hd.getIteration());
            assertEquals(length, hd.getOutputLength());
            try {
                hd.toBytes();
            } catch (InvalidPasswordCipherException ipce) {
                // normal
            } catch (Exception e) {
                fail("Unexpected exception : " + e.getMessage());
            }
        } catch (Exception e) {
            fail("Unexpected exception : " + e.getMessage());
        }
    }

    /**
     * Test constructor with invalid algorithm
     */
    @Test
    public void testConstructorInvalidIteration() {
        try {
            char plain[] = "password2".toCharArray();
            byte salt[] = "salt2".getBytes("UTF-8"); //cannot be autogenerated in order to compare with pre-hashed data.
            String algorithm = "PBKDF2WithHmacSHA1";
            int iteration = -1;
            int length = 512;
            HashedData hd = new HashedData(plain, algorithm, salt, iteration, length, null);
            assertTrue(Arrays.equals(hd.getPlain(), plain));
            assertTrue(Arrays.equals(hd.getSalt(), salt));
            assertEquals(algorithm, hd.getAlgorithm());
            assertEquals(iteration, hd.getIteration());
            try {
                hd.toBytes();
            } catch (InvalidPasswordCipherException ipce) {
                // normal
            } catch (Exception e) {
                fail("Unexpected exception : " + e.getMessage());
            }
        } catch (Exception e) {
            fail("Unexpected exception : " + e.getMessage());
        }
    }
}
