/*******************************************************************************
 * Copyright (c) 2008, 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ejs.container;

import java.lang.reflect.Method;
import java.rmi.RemoteException;
import java.util.Map;

import com.ibm.ejs.container.util.ExceptionUtil;
import com.ibm.websphere.ras.Tr;
import com.ibm.websphere.ras.TraceComponent;
import com.ibm.ws.ffdc.FFDCFilter;
import com.ibm.wsspi.ejbcontainer.WSEJBEndpointManager;

/**
 * Provides a mechanism for the WebServices component to request the
 * EJB Container to setup and tear down the EJB contexts and environments
 * associated with a WebService Endpoint request on an EJB. <p>
 * 
 * The EJB Container will provide a new instance of this wrapper for
 * every WebService Endpoint request. <p>
 * 
 * This wrapper allows the WebServices request on the EJB, as well as
 * invocations to the application handlers, to run in the EJB container
 * environment. <p>
 * 
 * This is not a traditional EJB Wrapper, in that it does NOT implement
 * all of the methods of the target interface (in this case, the
 * Service Endpoint Interface). Instead, it performs the following
 * two function:
 * 
 * 1) It provides SPIs to the WebServcies component to perform the
 * EJBContainer functions that a normal wrapper would, like calling
 * ejbPreInvoke and ejbPostInvoke.
 * 
 * 2) It provides an implementation of EJBWrapperBase that may be passed
 * through the normal ejbPreInvoke/ejbPostInvoke processing. An instance
 * of a wrapper is required for both of these methods.
 * 
 * The WebServices component will then use the SPIs provided on this
 * class to perform the same functions a normal wrapper would; using
 * reflections to actually invoke the method on the EJB instance. <p>
 * 
 * In the event that EJB Interceptors are present, the EJB instance will
 * NOT be returned to WebServices, but instead, a 'proxy/wrapper' object,
 * generated by JITDeploy, will be returned. WebServices will invoke it
 * using reflections, as if it were the EJB instance, and this 'proxy'
 * will invoke the EJB Interceptors. <p>
 **/
public class WSEJBWrapper extends EJSWrapperBase implements WSEJBEndpointManager
{
    private static final String CLASS_NAME = WSEJBWrapper.class.getName();

    private static final TraceComponent tc = Tr.register(WSEJBWrapper.class,
                                                         "EJBContainer",
                                                         "com.ibm.ejs.container.container");

    /**
     * Method this wrapper instance will be used to invoke.
     * 
     * Set from parameter on ejbPreInvoke.
     **/
    private Method ivMethod = null;

    /**
     * MethodContext, obtained in ejbPreInvoke, and held here for ejbPostInvoke.
     **/
    private EJSDeployedSupport ivMethodContext = null;

    /**
     * Method Index, determined in ejbPreInvoke, and held here for ejbPostInvoke.
     **/
    private int ivMethodIndex = Integer.MIN_VALUE;

    /**
     * Performs all EJB Container processing required prior to invoking
     * the specified EJB method. <p>
     * 
     * This method will establish the proper EJB contexts for the method
     * call; including the Security context, Transaction context,
     * and Naming context (java:comp/env), and the thread context
     * classloader. <p>
     * 
     * If this method fails (exception thrown) then the EJB contexts
     * and environment has not been properly setup, and no attempt
     * should be made to invoke the ejb, interceptors, or handlers. <p>
     * 
     * The method ejbPostInvoke MUST be called after calling this method;
     * to remove the EJB contexts from the current thread. This is true
     * even if this method fails with an exception. <p>
     * 
     * Note that the method arguments would normally be required for
     * EJB Container preInvoke processing, to properly determine JACC
     * security authorization, except the JACC specification, in section
     * 4.6.1.5 states the following: <p>
     * 
     * All EJB containers must register a PolicyContextHandler whose
     * getContext method returns an array of objects (Object[]) containing
     * the arguments of the EJB method invocation (in the same order as
     * they appear in the method signature) when invoked with the key
     * "javax.ejb.arguments". The context handler must return the value
     * null when called in the context of a SOAP request that arrived at
     * the ServiceEndpoint method interface. <p>
     * 
     * @param method A reflection Method object which provides the method
     *            name and signature of the EJB method that will be
     *            invoked. This may be a Method of either an EJB
     *            interface or the EJB implementation class.
     * @param context The MessageContext which should be returned when
     *            InvocationContext.getContextData() is called.
     * 
     * @return an object which may be invoked as though it were an
     *         instance of the EJB. It will be a wrapper/proxy object
     *         when there are EJB interceptors.
     * 
     * @throws RemoteException when a system level error occurs.
     **/
    public Object ejbPreInvoke(Method method, Map<String, Object> context)
                    throws RemoteException
    {
        // Validate the state, and input parameters
        if (ivMethod != null)
        {
            throw new IllegalStateException("WSEJBEndpointManager.ejbPreInvoke called previously : " +
                                            ivMethod.getName());
        }
        if (method == null)
        {
            throw new IllegalArgumentException("WSEJBEndpointManager.ejbPreInvoke requires a method");
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            Tr.entry(tc, "ejbPreInvoke : " + method.getName());

        Object[] args = null; // per JACC spec, section 4.6.1.5
        ivMethod = method;
        ivMethodIndex = getMethodIndex(method);

        // Get the EJS Deployed Support and save the message context.
        ivMethodContext = new EJSDeployedSupport();
        ivMethodContext.ivContextData = context; // d644886

        // Perform normal EJB Container preInvoke processing
        Object bean = container.EjbPreInvoke(this,
                                             ivMethodIndex,
                                             ivMethodContext,
                                             args);

        // If the bean has around invoke interceptors on any method, then
        // instead of returning the bean instance, a JIT Deploy generated
        // proxy object is returned, which knows how to invoke the around
        // invoke interceptors, as needed.                                 d497921
        if (bmd.ivWebServiceEndpointProxyClass != null)
        {
            try
            {
                WSEJBProxy proxy = (WSEJBProxy) bmd.ivWebServiceEndpointProxyClass.newInstance();
                proxy.ivContainer = container;
                proxy.ivMethodContext = ivMethodContext;
                proxy.ivEjbInstance = bean;
                bean = proxy;
            } catch (Throwable ex)
            {
                // This should never occur... only if JITDeploy generated a class
                // that could be loaded, but not used. No sense logging an error,
                // this must be reported to service for a fix.
                FFDCFilter.processException(ex, CLASS_NAME + ".ejbPreInvoke", "192", this);
                throw ExceptionUtil.EJBException
                                ("Failed to create proxy for Web service endpoint", ex);
            }
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            Tr.exit(tc, "ejbPreInvoke : " + ivMethod.getName() + " : " +
                        bean.getClass().getName());

        return bean;
    }

    /**
     * Performs all EJB Container processing required to remove the EJB contexts
     * established by ejbPreInvoke. <p>
     * 
     * This method is intended to be called after calling ejbPreInvoke and
     * after the EJB method has completed. This method MUST be called anytime
     * ejbPreInvoke has been called, even if ejbPreInvoke fails with an
     * exception. <p>
     * 
     * If an exception occurs during ejbPostInvoke, or setException was called
     * with a non-application exception, then the EJB Container will map the
     * exception to an appropriate RemoteException as required by the EJB
     * Specification, and throw the mapped exception. <p>
     * 
     * @throws RemoteException when a system level error has occurred.
     **/
    public void ejbPostInvoke()
                    throws RemoteException
    {
        // Validate the state
        if (ivMethod == null)
        {
            throw new IllegalStateException("WSEJBEndpointManager.ejbPreInvoke must be called first.");
        }
        if (ivMethodContext == null)
        {
            if (ivMethodIndex != Integer.MIN_VALUE)
            {
                throw new IllegalStateException("WSEJBEndpointManager.ejbPostInvoke already called.");
            }
            // ejbPreInvoke called but failed; allow ejbPostInvoke
            return;
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            Tr.entry(tc, "ejbPostInvoke : " + ivMethod.getName());

        // Perform normal EJB Container postInvoke processing
        try
        {
            container.postInvoke(this, ivMethodIndex, ivMethodContext);
        } finally
        {
            ivMethodContext = null;

            if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
                Tr.exit(tc, "ejbPostInvoke : " + ivMethod.getName());
        }
    }

    /**
     * Notifies the EJB Container that an exception occurred during
     * processing of the endpoint method, and allows the EJB Container
     * to map that exception to the appropriate exception required
     * by the EJB Specification. <p>
     * 
     * This method should be called for any exception that occurs
     * when invoking the WebService Handlers, EJB Interceptors, or
     * the EJB method itself. <p>
     * 
     * Generally, if the exception provided is an application exception,
     * then it will be returned, unmapped. If it is a system exception,
     * then it will be mapped per the Exception Handling chapter of the
     * EJB Specification. Note that RuntimeExceptions may be considered
     * application exception beginning in EJB 3.0. <p>
     * 
     * Calling setException may or may not result in the transaction
     * being marked for rollback or rolled back. System exceptions
     * will always result in rollback. Application exceptions may
     * or may not, depending on customer configuration. <p>
     * 
     * @param ex throwable being reported to the EJB Container.
     * 
     * @return either the exception passed in, if it is an application
     *         exception, or the appropriate exception as required
     *         by the EJB Specification.
     **/
    // d503197
    public Throwable setException(Throwable ex)
    {
        // Validate the state
        if (ivMethod == null)
        {
            throw new IllegalStateException("WSEJBEndpointManager.ejbPreInvoke must be called first.");
        }
        if (ivMethodContext == null)
        {
            if (ivMethodIndex != Integer.MIN_VALUE)
            {
                throw new IllegalStateException("WSEJBEndpointManager.ejbPostInvoke already called.");
            }
            // ejbPreInvoke called but failed; allow setException
            return ex;
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            Tr.entry(tc, "setException : " + ex.getClass().getName());

        Throwable mappedException = null;
        ExceptionMappingStrategy exStrategy = ivMethodContext.getExceptionMappingStrategy();

        // -----------------------------------------------------------------------
        // First, determine if this is a 'checked' application exception.
        //
        // Where 'checked' means it is on the throws clause, and application
        // (per EJB spec) means it must be a subclass of Exception (including
        // RuntimeException), but not a subclass of RemoteException.
        //
        // Checked application exceptions are not mapped, so just set it on
        // the method context (EJSDeployedSupport) and return the same exception.
        // -----------------------------------------------------------------------

        if (ex instanceof Exception &&
            !(ex instanceof RemoteException))
        {
            Class<?> exceptionClass = ex.getClass();
            Class<?>[] checkedExceptions = ivMethod.getExceptionTypes();

            // interate checked exception array and look for a match.
            for (Class<?> checkedClass : checkedExceptions)
            {
                if (checkedClass.isAssignableFrom(exceptionClass))
                {
                    // This is a checked exception, so call setCheckedException
                    // to record this fact, and return the exception - no mapping.
                    exStrategy.setCheckedException(ivMethodContext, (Exception) ex);
                    if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
                        Tr.exit(tc, "setException : " + ex.getClass().getName());
                    return ex;
                }
            }
        }

        // -----------------------------------------------------------------------
        // This is an 'unchecked' exception - perform mapping as necessary.
        //
        // Note that this may still be an application exception, as beginning
        // with EJB 3.0, RuntimeExceptions may also be application exceptions.
        //
        // Set the exception on the method context (EJSDeployedSupport), which
        // will determine if this is an application exception or not, perform
        // any necessary mapping, and allow postInvoke to properly decide
        // whether to rollback the transaction or not.
        // -----------------------------------------------------------------------

        mappedException = exStrategy.setUncheckedException(ivMethodContext, ex);

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            Tr.exit(tc, "setException : " + mappedException.getClass().getName());

        return mappedException;
    }

    /**
     * Returns the method index into the array of methods for the
     * WebService Endpoint interface. <p>
     * 
     * The method index is used by ejbPreInvoke to locate the
     * correct method level information, such as transaction
     * attributes. <p>
     * 
     * This is the value that would normally be hard coded
     * into a generated (EJBDeploy or JITDeploy) wrapper. <p>
     * 
     * @throws EJBException when the specified method is not
     *             present on the WebService Endpoint interface.
     **/
    private int getMethodIndex(Method method)
    {
        for (int i = 0; i < methodInfos.length; ++i)
        {
            if (methodInfos[i].methodsMatch(method)) // d517824
            {
                return i;
            }
        }

        // Log the error and throw meaningful exception.
        Tr.error(tc, "WS_ENDPOINT_METHOD_MISSING_CNTR0178E",
                 new Object[] { method.getName(),
                               bmd.j2eeName.getComponent(),
                               bmd.j2eeName.getModule(),
                               bmd.j2eeName.getApplication() });
        throw ExceptionUtil.EJBException
                        ("Configured Web service endpoint method " + method.getName() +
                         " is not implemented by the " + bmd.j2eeName.getComponent() +
                         " bean in the " + bmd.j2eeName.getModule() + " module of the " +
                         bmd.j2eeName.getApplication() + " application.", null);
    }

    /**
     * Confirms that the specified methods are implemented by the specified
     * bean, and updates the Web Service Endpoint EJBMethodInfo array in
     * BeanMetaData to only contain the EJBMethodInfos corresponding to the
     * specified methods. And returns true if any of the methods have
     * around invoke interceptors defined. <p>
     * 
     * Since Web Services component now processes all Web Services related
     * metadata, EJB Container will not know the list of Web Service Endpoint
     * methods (unless an interface is specified in ejb-jar.xml) until the
     * WebServices component calls createWebSerivceEndpointManager().
     * Thus, EJB Container must assume that all methods on the bean may be
     * WebService Endpoint methods. <p>
     * 
     * This method is intended to be used by createWebSerivceEndpointManager()
     * to validate the methods passed by the Web Services component and update
     * the EJB Container internals with the known set of methods. <p>
     * 
     * @param bmd bean metadata for the endpoint being resolved.
     * @param endpointMethods list of Web Service Endpoint methods.
     * 
     * @return true if any of the methods have around invoke interceptors.
     * 
     * @throws EJBConfigurationException if any of the specified endpoint
     *             methods are not implemented by the bean.
     **/
    // d497921
    static boolean resolveWebServiceEndpointMethods(BeanMetaData bmd,
                                                    Method[] endpointMethods)
                    throws EJBConfigurationException
    {
        if (endpointMethods == null || endpointMethods.length == 0)
        {
            throw new EJBConfigurationException("Web service endpoint configurd with no methods : " + bmd.j2eeName);
        }

        if (bmd.wsEndpointMethodInfos == null)
        {
            throw new EJBConfigurationException("Web service endpoint is only allwed for stateless session " +
                                                "beans, and a Web service endpoint interface must be " +
                                                "configured in ejb-jar.xml if the module level is prior " +
                                                "to 3.0 : " + bmd.j2eeName);
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            Tr.entry(tc, "resolveWebServiceEndpointMethods : " + bmd.j2eeName);

        boolean hasAroundInvoke = false;
        int numMethodInfos = bmd.wsEndpointMethodInfos.length;
        int numEndpointMethods = endpointMethods.length;
        String[] endpointMethodNames = new String[numEndpointMethods];
        EJBMethodInfoImpl[] endpointMethodInfos = new EJBMethodInfoImpl[numEndpointMethods];

        for (int i = 0; i < numEndpointMethods; ++i)
        {
            Method method = endpointMethods[i];
            for (int j = 0; j < numMethodInfos; ++j)
            {
                EJBMethodInfoImpl methodInfo = bmd.wsEndpointMethodInfos[j];
                if (methodInfo.methodsMatch(method)) // d517824
                {
                    endpointMethodInfos[i] = methodInfo;
                    endpointMethodNames[i] = method.getName();
                    if (methodInfo.ivAroundInterceptors != null) // F743-17763.1
                        hasAroundInvoke = true;

                    // When customer has used generics, and the methodinfos have
                    // been built from all methods on the bean, then there will be
                    // a methodinfo for both the bridge and target methods.
                    // Keep looking until there is a match with a 'bridge' method.
                    // If one is not found, then no generics and the first one found
                    // is then the correct one and will be used.              d540438
                    if (methodInfo.getBridgeMethod() != null)
                        break;
                }
            }
            if (endpointMethodInfos[i] == null)
            {
                // Log the error and throw meaningful exception.
                Tr.error(tc, "WS_ENDPOINT_METHOD_MISSING_CNTR0178E",
                         new Object[] { method.getName(),
                                       bmd.j2eeName.getComponent(),
                                       bmd.j2eeName.getModule(),
                                       bmd.j2eeName.getApplication() });
                throw new EJBConfigurationException("Configured Web service endpoint method " + method.getName() +
                                                    " is not implemented by the " + bmd.j2eeName.getComponent() +
                                                    " bean in the " + bmd.j2eeName.getModule() + " module of the " +
                                                    bmd.j2eeName.getApplication() + " application.");
            }
        }

        bmd.wsEndpointMethodInfos = endpointMethodInfos;
        bmd.wsEndpointMethodNames = endpointMethodNames;

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            Tr.exit(tc, "resolveWebServiceEndpointMethods : " + bmd.j2eeName +
                        " : aroundInvoke = " + hasAroundInvoke);

        return hasAroundInvoke;
    }

}
