/*******************************************************************************
 * Copyright (c) 2009, 2018 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.ejbcontainer.interceptor.aroundTimeout_ann.ejb;

import static com.ibm.ws.ejbcontainer.interceptor.aroundTimeout_ann.ejb.AdvancedInterface.AUTO_TIMER_INFO;

import java.util.concurrent.CountDownLatch;
import java.util.logging.Logger;

import javax.annotation.Resource;
import javax.ejb.Local;
import javax.ejb.Schedule;
import javax.ejb.Singleton;
import javax.ejb.Timeout;
import javax.ejb.Timer;
import javax.ejb.TimerConfig;
import javax.ejb.TimerService;
import javax.interceptor.AroundTimeout;
import javax.interceptor.Interceptors;
import javax.interceptor.InvocationContext;

@Singleton()
@Local(AdvancedInterface.class)
@Interceptors({ CL1Interceptor.class, CL2Interceptor.class })
public class AdvancedAroundTimeoutBean extends SuperAdvancedAroundTimeout {
    private static final String CLASS_NAME = AdvancedAroundTimeoutBean.class.getName();
    private static final Logger svLogger = Logger.getLogger(CLASS_NAME);

    private static CountDownLatch svTimerLatch;
    private static final CountDownLatch svAutoTimerLatch = new CountDownLatch(1);

    @Resource
    private TimerService ivTimerService;

    public CountDownLatch createSingleActionTimer(String info) {
        svTimerLatch = new CountDownLatch(1);
        TimerConfig config = new TimerConfig(info, false);
        Timer singleAction = ivTimerService.createSingleActionTimer(5, config);
        svLogger.info("Created single action timer with info + " + info + " : " + singleAction);

        return svTimerLatch;
    }

    public CountDownLatch getAutoTimerLatch() {
        return svAutoTimerLatch;
    }

    @AroundTimeout
    private Object aroundTimeout(InvocationContext c) throws Exception {
        svLogger.info("--> Entered " + CLASS_NAME + ".aroundTimeout");
        try {
            Timer t = (Timer) c.getTimer();
            svLogger.info("--> Timer t = " + t);
            String eventTag = "::" + this + ".aroundTimeout:" + c.getMethod() + "," + t.getInfo();

            svLogger.info("--> eventTag = " + eventTag);
            TimerData.addIntEvent(t, eventTag);

            return c.proceed();
        } finally {
            svLogger.info("<-- Exiting " + CLASS_NAME + ".aroundTimeout");
        }
    }

    @Interceptors({ ML1Interceptor.class, ML2Interceptor.class })
    @Timeout
    public void timeoutMethod(Timer t) {
        svLogger.info("--> Entered " + CLASS_NAME + ".timeoutMethod");
        try {
            svLogger.info("--> Timer t = " + t);
            String infoKey = (String) t.getInfo();
            svLogger.info("--> infoKey = " + infoKey);
            TimerData td = TimerData.svIntEventMap.get(infoKey);
            svLogger.info("--> svIntEventMap = " + TimerData.svIntEventMap);
            svLogger.info("--> td = " + td);

            String eventTag = "::" + this + ".timeoutMethod:" + t.getInfo();
            svLogger.info("--> eventTag = " + eventTag);
            TimerData.addIntEvent(t, eventTag);

            td.setIsFired(true);
            svTimerLatch.countDown();
        } finally {
            svLogger.info("<-- Exiting " + CLASS_NAME + ".timeoutMethod");
        }
    }

    @Interceptors({ ML1Interceptor.class, ML2Interceptor.class })
    @Schedule(second = "*", minute = "*", hour = "*", info = AUTO_TIMER_INFO, persistent = false)
    public void autoTimeoutMethod(Timer t) {
        svLogger.info("--> Entered " + CLASS_NAME + ".autoTimeoutMethod");
        try {
            svLogger.info("--> Timer t = " + t);
            String infoKey = (String) t.getInfo();
            svLogger.info("--> infoKey = " + infoKey);

            TimerData td = TimerData.svIntEventMap.get(infoKey);
            svLogger.info("--> svIntEventMap = " + TimerData.svIntEventMap);
            svLogger.info("--> td = " + td);

            String eventTag = "::" + this + ".autoTimeoutMethod:" + t.getInfo();
            svLogger.info("--> eventTag = " + eventTag);
            TimerData.addIntEvent(t, eventTag);

            td.setIsFired(true);
            svAutoTimerLatch.countDown();
        } finally {
            svLogger.info("--> Cancelling timer...");
            t.cancel();
            svLogger.info("<-- Exiting " + CLASS_NAME + ".autoTimeoutMethod");
        }
    }
}
