/**
 * Copyright (c) 2019 Red Hat, Inc.
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at:
 *
 *     https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   Red Hat, Inc. - initial API and implementation
 */
package org.eclipse.jkube.kit.build.api.assembly;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.jkube.kit.common.AssemblyFileEntry;
import org.eclipse.jkube.kit.common.archive.ArchiveCompression;
import org.eclipse.jkube.kit.common.archive.JKubeTarArchiver;
import org.eclipse.jkube.kit.common.util.FileUtil;

public class JKubeBuildTarArchiver {

    public static final String ARCHIVE_FILE_NAME = "docker-build.";
    private final Map<File, String> filesToIncludeNameMap = new HashMap<>();
    private final Map<File, String> fileModeMap = new HashMap<>();
    private final List<String> filesNamesToExclude = new ArrayList<>();

    public void includeFile(File inputFile, String destinationFileName) {
        filesToIncludeNameMap.put(inputFile, destinationFileName);
    }

    public void setFileMode(AssemblyFileEntry assemblyFileEntry) {
        fileModeMap.put(assemblyFileEntry.getDest(), assemblyFileEntry.getFileMode());
    }

    public void excludeFile(String inputFilePath) {
        filesNamesToExclude.add(inputFilePath);
    }

    public File createArchive(File inputDirectory, BuildDirs buildDirs, ArchiveCompression compression) throws IOException {
        File outputFile = new File(buildDirs.getTemporaryRootDirectory(), ARCHIVE_FILE_NAME + (compression.equals(ArchiveCompression.none) ? "tar" : compression.getFileSuffix()));
        List<File> files = FileUtil.listFilesAndDirsRecursivelyInDirectory(inputDirectory);

        if (!filesToIncludeNameMap.isEmpty()) {
            for (Map.Entry<File, String> entry : filesToIncludeNameMap.entrySet()) {
                File srcFile = entry.getKey();
                String targetFileName = entry.getValue();

                // 1. Check whether nested directory is there, if not create it
                File parentDirectory = new File(targetFileName).getParentFile();
                if (parentDirectory != null) {
                    FileUtil.createDirectory(new File(inputDirectory, parentDirectory.getPath()));
                }
                File targetFile = new File(inputDirectory, targetFileName);
                if (!srcFile.equals(targetFile)) {
                    FileUtil.copy(srcFile, targetFile);
                    files.add(targetFile);
                }
            }
        }

        List<File> fileListToAddInTarball = new ArrayList<>();
        for (File currentFile : files) {
            if (filesNamesToExclude.contains(currentFile.getName())) {
                continue;
            }
            fileListToAddInTarball.add(currentFile);
        }

        return JKubeTarArchiver.createTarBall(outputFile, inputDirectory, fileListToAddInTarball, fileModeMap, compression);
    }
}