package org.jboss.brmsbpmsuite.patching.systemproperty;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.Text;

import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.util.Map;

public class SystemPropertyPatcher {

    private final Document patchedDocument;
    private final Element systemProperties;
    private final Map<String, Node> systemPropertiesIndex;

    public SystemPropertyPatcher(Document baseDocument) throws ParserConfigurationException {
        patchedDocument = duplicateBase(baseDocument);
        systemProperties = getOrCreateSystemProperties();
        systemPropertiesIndex = SystemPropertyLoader.load(patchedDocument);
    }

    private Document duplicateBase(Document baseDocument) throws ParserConfigurationException {
        Node newDocumentElement = baseDocument.getDocumentElement().cloneNode(true);
        Document newDocument = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
        newDocument.appendChild(newDocument.adoptNode(newDocumentElement));
        return newDocument;
    }

    private Element getOrCreateSystemProperties() {
        Element systemProperties = (Element) patchedDocument.getElementsByTagName(SystemPropertyNodeName.SYSTEM_PROPERTIES.text).item(0);
        if (systemProperties == null) {
            systemProperties = patchedDocument.createElement(SystemPropertyNodeName.SYSTEM_PROPERTIES.text);
            Element extensions = (Element) patchedDocument.getElementsByTagName(SystemPropertyNodeName.EXTENSIONS.text).item(0);
            // insert <system-properties/> after <extensions/>
            patchedDocument.getFirstChild().insertBefore(systemProperties, extensions.getNextSibling());
        }
        return systemProperties;
    }

    public Document apply(Document diffDocument) {
        Map<String, Node> diffProperties = SystemPropertyLoader.load(diffDocument);
        for (Map.Entry<String, Node> diffPropertiesEntry : diffProperties.entrySet()) {
            String diffPropertyName = diffPropertiesEntry.getKey();
            Node diffPropertyValue = diffPropertiesEntry.getValue();
            if (systemPropertiesIndex.containsKey(diffPropertyName)) {
                applyToExistingProperty((Element) diffPropertyValue);
            } else {
                applyToNonExistingProperty((Element) diffPropertyValue);
            }
        }

        return patchedDocument;
    }

    private void applyToExistingProperty(Element diffProperty) {
        String action = diffProperty.getAttribute(SystemPropertyNodeName.ACTION.text);
        switch (SystemPropertyAction.from(action)) {
            case ADD:
            case UPDATE:
                updateProperty(diffProperty);
                break;

            case REMOVE:
                removeProperty(diffProperty);
        }
    }

    private void applyToNonExistingProperty(Element diffProperty) {
        String action = diffProperty.getAttribute(SystemPropertyNodeName.ACTION.text);
        switch (SystemPropertyAction.from(action)) {
            case ADD:
            case UPDATE:
                addProperty(diffProperty);
                break;
        }
    }

    private void addProperty(Element added) {
        Element adopted = (Element) patchedDocument.adoptNode(added.cloneNode(true));
        adopted.removeAttribute(SystemPropertyNodeName.ACTION.text);

        Node firstChild = systemProperties.getFirstChild();
        Node lastChild = systemProperties.getLastChild();

        if (isBlankTextNode(firstChild) && isBlankTextNode(lastChild)) {
            systemProperties.insertBefore(adopted, lastChild);
            systemProperties.insertBefore(firstChild.cloneNode(false), adopted);
        } else {
            systemProperties.appendChild(adopted);
        }
    }

    private void updateProperty(Element updated) {
        String name = updated.getAttribute(SystemPropertyNodeName.NAME.text);
        String value = updated.getAttribute(SystemPropertyNodeName.VALUE.text);
        Element updatedProperty = (Element) systemPropertiesIndex.get(name);
        updatedProperty.setAttribute(SystemPropertyNodeName.VALUE.text, value);
    }

    private void removeProperty(Element removed) {
        String name = removed.getAttribute(SystemPropertyNodeName.NAME.text);
        Node removedProperty = systemPropertiesIndex.get(name);
        while(isBlankTextNode(removedProperty.getNextSibling())) {
            systemProperties.removeChild(removedProperty.getNextSibling());
        }
        systemProperties.removeChild(removedProperty);
    }

    private boolean isBlankTextNode(Node node) {
        return node instanceof Text && node.getTextContent().replaceAll("\\W", "").length() == 0;
    }

}
