package io.vertx.ext.auth.test.oauth2;

import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.HttpServer;
import io.vertx.core.json.JsonObject;
import io.vertx.ext.auth.User;
import io.vertx.ext.auth.authorization.PermissionBasedAuthorization;
import io.vertx.ext.auth.impl.http.SimpleHttpClient;
import io.vertx.ext.auth.oauth2.OAuth2Auth;
import io.vertx.ext.auth.oauth2.OAuth2FlowType;
import io.vertx.ext.auth.oauth2.OAuth2Options;
import io.vertx.ext.auth.oauth2.authorization.ScopeAuthorization;
import io.vertx.test.core.VertxTestBase;
import org.junit.Test;

import java.io.UnsupportedEncodingException;
import java.util.concurrent.CountDownLatch;

public class OAuth2IntrospectTest extends VertxTestBase {

  // according to RFC
  private static final JsonObject fixtureIntrospect = new JsonObject(
    "{" +
      "  \"active\": true," +
      "  \"scope\": \"scopeA scopeB\"," +
      "  \"client_id\": \"client-id\"," +
      "  \"username\": \"username\"," +
      "  \"token_type\": \"bearer\"," +
      "  \"exp\": 99999999999," +
      "  \"iat\": 7200," +
      "  \"nbf\": 7200" +
      "}");

  // according to Google
  private static final JsonObject fixtureGoogle = new JsonObject(
    "{" +
      "  \"audience\": \"8819981768.apps.googleusercontent.com\"," +
      "  \"user_id\": \"123456789\"," +
      "  \"scope\": \"profile email\"," +
      "  \"expires_in\": 436" +
      "}");

  // according to Keycloak
  private static final JsonObject fixtureKeycloak = new JsonObject(
    "{" +
      "  \"active\": true," +
      "  \"exp\": 99999999999," +
      "  \"iat\": 1465313839," +
      "  \"aud\": \"hello-world-authz-service\",\n" +
      "  \"nbf\": 0" +
      "}");

  private static final String token = "eyJhbGciOiJub25lIiwidHlwIjoiSldUIn0.eyJhdXRob3JpemF0aW9uIjp7InBlcm1pc3Npb25zIjpbeyJyZXNvdXJjZV9zZXRfaWQiOiJkMmZlOTg0My02NDYyLTRiZmMtYmFiYS1iNTc4N2JiNmUwZTciLCJyZXNvdXJjZV9zZXRfbmFtZSI6IkhlbGxvIFdvcmxkIFJlc291cmNlIn1dfSwianRpIjoiZDYxMDlhMDktNzhmZC00OTk4LWJmODktOTU3MzBkZmQwODkyLTE0NjQ5MDY2Nzk0MDUiLCJleHAiOjk5OTk5OTk5OTksIm5iZiI6MCwiaWF0IjoxNDY0OTA2NjcxLCJzdWIiOiJmMTg4OGY0ZC01MTcyLTQzNTktYmUwYy1hZjMzODUwNWQ4NmMiLCJ0eXAiOiJrY19ldHQiLCJhenAiOiJoZWxsby13b3JsZC1hdXRoei1zZXJ2aWNlIn0";

  private static final JsonObject oauthIntrospect = new JsonObject()
    .put("token", token);

  private OAuth2Auth oauth2;
  private HttpServer server;
  private JsonObject config;
  private JsonObject fixture;

  private final OAuth2Options oauthConfig = new OAuth2Options()
    .setFlow(OAuth2FlowType.AUTH_CODE)
    .setClientId("client-id")
    .setClientSecret("client-secret")
    .setSite("http://localhost:8080")
    .setIntrospectionPath("/oauth/introspect");

  @Override
  public void setUp() throws Exception {
    super.setUp();
    oauth2 = OAuth2Auth.create(vertx, oauthConfig);

    final CountDownLatch latch = new CountDownLatch(1);

    server = vertx.createHttpServer().requestHandler(req -> {
      if (req.method() == HttpMethod.POST && "/oauth/introspect".equals(req.path())) {
        req.setExpectMultipart(true).bodyHandler(buffer -> {
          try {
            JsonObject body = SimpleHttpClient.queryToJson(buffer);
            assertEquals(config.getString("token"), body.getString("token"));
            // conditional test for token_type_hint
            if (config.containsKey("token_type_hint")) {
              assertEquals(config.getString("token_type_hint"), body.getString("token_type_hint"));
            }
          } catch (UnsupportedEncodingException e) {
            fail(e);
          }
          req.response().putHeader("Content-Type", "application/json").end(fixture.encode());
        });
      } else if (req.method() == HttpMethod.POST && "/oauth/tokeninfo".equals(req.path())) {
        req.setExpectMultipart(true).bodyHandler(buffer -> {
          try {
            assertEquals(config, SimpleHttpClient.queryToJson(buffer));
          } catch (UnsupportedEncodingException e) {
            fail(e);
          }
          req.response().putHeader("Content-Type", "application/json").end(fixture.encode());
        });
      } else {
        req.response().setStatusCode(400).end();
      }
    }).listen(8080, ready -> {
      if (ready.failed()) {
        throw new RuntimeException(ready.cause());
      }
      // ready
      latch.countDown();
    });

    latch.await();
  }

  @Override
  public void tearDown() throws Exception {
    server.close();
    super.tearDown();
  }

  @Test
  public void introspectAccessToken() {
    config = oauthIntrospect;
    fixture = fixtureIntrospect;
    oauth2.authenticate(new JsonObject().put("access_token", token).put("token_type", "Bearer"), res -> {
      if (res.failed()) {
        fail(res.cause().getMessage());
      } else {
        User token2 = res.result();
        assertNotNull(token2);
        JsonObject principal = token2.principal().copy();

        // clean time specific value
        principal.remove("expires_at");
        principal.remove("access_token");

        final JsonObject assertion = fixtureIntrospect.copy();

        assertEquals(assertion.getMap(), principal.getMap());

        ScopeAuthorization.create(" ").getAuthorizations(token2, res0 -> {
          if (res0.failed()) {
            fail(res0.cause().getMessage());
          } else {
            if (PermissionBasedAuthorization.create("scopeB").match(token2)) {
              testComplete();
            } else {
              fail("Should be allowed");
            }
          }
        });
      }
    });
    await();
  }

  @Test
  public void introspectAccessTokenGoogleWay() {
    config = oauthIntrospect;
    fixture = fixtureGoogle;
    oauth2.authenticate(new JsonObject().put("access_token", token).put("token_type", "Bearer"), res -> {
      if (res.failed()) {
        fail(res.cause().getMessage());
      } else {
        User token = res.result();
        assertNotNull(token);
        // make a copy because later we need to original data
        JsonObject principal = token.principal().copy();

        // clean up control
        final JsonObject assertion = fixtureGoogle.copy();

        assertEquals(assertion.getMap(), principal.getMap());

        ScopeAuthorization.create(" ").getAuthorizations(token, res0 -> {
          if (res0.failed()) {
            fail(res0.cause().getMessage());
          } else {
            if (PermissionBasedAuthorization.create("profile").match(token)) {
              // Issue #142

              // the test is a replay of the same test so all checks have
              // been done above.

              // the replay shows that the api can be used from the user object
              // directly too
              oauth2.authenticate(token.principal().put("access_token", OAuth2IntrospectTest.token).put("token_type", "Bearer"), v -> {
                if (v.failed()) {
                  fail(v.cause());
                } else {
                  testComplete();
                }
              });
            } else {
              fail("Should be allowed");
            }
          }
        });
      }
    });
    await();
  }

  @Test
  public void introspectAccessTokenKeyCloakWay() {
    config = oauthIntrospect;
    fixture = fixtureKeycloak;
    oauth2.authenticate(new JsonObject().put("access_token", token).put("token_type", "Bearer"), res -> {
      if (res.failed()) {
        fail(res.cause());
      } else {
        User token = res.result();
        assertNotNull(token);
        assertNotNull(token.principal());
        testComplete();
      }
    });
    await();
  }
}
