package com.izforge.izpack.util;

import org.jboss.aesh.complete.CompleteOperation;
import org.jboss.aesh.complete.Completion;
import org.jboss.aesh.console.*;
import org.jboss.aesh.console.Console;
import org.jboss.aesh.console.helper.InterruptHook;
import org.jboss.aesh.console.settings.Settings;
import org.jboss.aesh.console.settings.SettingsBuilder;
import org.jboss.aesh.edit.actions.Action;
import java.io.File;
import java.io.IOException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Shell.java replacement using AEsh
 * Created by thauser on 8/20/14.
 */
public class Shell {

    private Console console;
    private static Shell SHELL;
    private boolean directoryField;
    private boolean disableFileCompletion;
    public volatile boolean lineRead = false;
    private volatile String input;

    class IzPackConsoleCallback extends AeshConsoleCallback {
        @Override
        public int execute(ConsoleOperation output) throws InterruptedException {
            input = output.getBuffer();
            lineRead = true;
            return 0;
        }
    }

    private Shell() {
        console = new Console(buildSettings());
        console.addCompletion(fileCompleter);
        console.setConsoleCallback(new IzPackConsoleCallback());
        console.start();
    }

    private Settings buildSettings() {
        SettingsBuilder settings = new SettingsBuilder();
        settings.readInputrc(false);
        settings.disableHistory(true);
        settings.enableAlias(false);
        settings.enableExport(false);
        settings.parseOperators(false);
        if (System.getProperty("os.name").toLowerCase().startsWith("windows")) {
            String directConsole = System.getProperty("installer.direct.console");
            if (directConsole != null && directConsole.equals("true")) {
                settings.ansi(false);
            }
        }
        setDisableFileCompletion(true);
        setDirectoryField(false);

        settings.interruptHook(new InterruptHook() {
            @Override
            public void handleInterrupt(Console console, Action action) {
                if (console != null){
                    console.stop();
                }
                Thread.currentThread().interrupt();
                System.exit(1);
            }
        });

        return settings.create();
    }

    public static Shell getInstance() {
        if (SHELL == null) {
            SHELL = new Shell();
        }
        return SHELL;
    }

    public static void shutdown() {
        if (SHELL != null) {
            SHELL.stopShell();
        }
    }

    public String getInput() {
        return getInput(false);
    }


    public String getInput(boolean raw) {
        while (!lineRead) {
            try {
                Thread.sleep(10);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        lineRead = false;
        if (raw) {
            return input;
        } else {
            return input.trim();
        }
    }

    public String getInputInterruptable(boolean raw) throws InterruptedException {
        while (!lineRead){
            Thread.sleep(10);
        }
        lineRead = false;
        if (raw)
            return input;
        else
            return input.trim();
    }

    /**
     * Get a folder location from the user.
     * Supports autocomplete features.
     * We trim off any trailing whitepsace.
     * The check for line.length() > 1, is an exception to leave the slash for the root directory.
     */
    public String getLocation(boolean directoryField) {
        String line;
        setDirectoryField(directoryField);
        setDisableFileCompletion(false);
        line = getInput();
        line = line.replace("~", System.getProperty("user.home"));
        return line.trim();
    }

    /**
     * Get a password from the user.
     * Currently the ConsoleReader does not support hiding the password being entered.
     */
    public String getPassword() {
        Prompt oldPrompt = console.getPrompt();
        Prompt passwordPrompt = new Prompt("", '*');
        String line;
        setDisableFileCompletion(true);
        console.setPrompt(passwordPrompt);
        line = getInput();
        console.setPrompt(oldPrompt);
        return line;
    }

    /**
     * Get a single character from the user
     *
     * @return A character that the user has entered.
     */
    public char getChar() throws IOException {
        String line;
        setDisableFileCompletion(true);
        line = getInput();
        return line.toLowerCase().charAt(0);
    }

    public void stopShell() {
        console.stop();
    }

    private void setDirectoryField(boolean isDirectory) {
        this.directoryField = isDirectory;
    }

    private void setDisableFileCompletion(boolean disableFileCompletion) {
        this.disableFileCompletion = disableFileCompletion;
    }

    private boolean getDirectoryField() {
        return directoryField;
    }

    private boolean getDisableFileCompletion() {
        return disableFileCompletion;
    }

    Completion fileCompleter = new Completion() {
        @Override
        public void complete(CompleteOperation co) {

            if (getDisableFileCompletion())
                return;

            String originalEntry = co.getBuffer().trim();
            String subbedEntry = originalEntry;
            if (subbedEntry.startsWith("~")) {
                subbedEntry = subbedEntry.replaceFirst("~", System.getProperty("user.home"));
            }
            String rest = getLastPathSection(subbedEntry);
            File currFile = new File(subbedEntry);

            if (!subbedEntry.isEmpty() && (currFile.isDirectory() || currFile.getParent() != null)) {

                File[] filesInDir = null;
                PrefixFileFilter fileFilter = new PrefixFileFilter(rest, getDirectoryField());

                // handle case of root directory, ie "/"
                if (currFile.getParentFile() == null && currFile.isDirectory()) {

                    if (System.getProperty("os.name").startsWith("Windows")) {
                        Pattern root = Pattern.compile("^[A-Z]:$");
                        Matcher rootMatcher = root.matcher(currFile.getPath());
                        if (rootMatcher.matches()) {
                            co.addCompletionCandidate("\\");
                        } else {
                            filesInDir = currFile.listFiles(fileFilter);
                        }
                    } else {
                        filesInDir = currFile.listFiles(fileFilter);
                    }
                }
                // handle case of paths that have a parent dir but aren't a dir or file them selves, "/home/username/not_fini"
                else if (!currFile.exists() && currFile.getParentFile() != null) {
                    if (System.getProperty("os.name").startsWith("Windows") && !currFile.getPath().contains(System.getProperty("file.separator"))) {
                        filesInDir = null;
                    } else {
                        filesInDir = currFile.getParentFile().listFiles(fileFilter);
                    }
                }
                // if it is either a file or directory and not the root dir
                else if (currFile.exists()) {
                    // This means that its a file or dir and the only one that is possible at this point, so look for things in this or file now
                    if (rest.length() > 0) {
                        filesInDir = currFile.getParentFile().listFiles(fileFilter);
                    }
                    // Means that it ended in slash meaning a new dir
                    else {
                        filesInDir = currFile.listFiles(fileFilter);
                    }
                }

                if (filesInDir != null) {
                    for (File f : filesInDir) {
                        String candidate = f.getAbsolutePath();
                        candidate = getLastPathSection(candidate);
                        if (f.isDirectory()) {
                            candidate = candidate + System.getProperty("file.separator");
                        }
                        co.addCompletionCandidate(candidate);
                    }
                }
            }

            co.setOffset(originalEntry.length() - rest.length());
            co.doAppendSeparator(false);
        }
    };

    private String getLastPathSection(String path) {
        if (path.endsWith(System.getProperty("file.separator"))) {
            return "";
        }
        String fileSeparator = System.getProperty("file.separator");
        // Because \\ is not a valid regex
        if (fileSeparator.equals("\\")) {
            fileSeparator = fileSeparator.concat(fileSeparator);
        }
        String[] pathSegments = path.split(fileSeparator);
        String rest = "";
        if (pathSegments.length > 1) {
            rest = pathSegments[pathSegments.length - 1];
        }
        return rest;
    }
}
