package org.jboss.brmsbpmsuite.patching.client;

import com.google.common.collect.Lists;
import org.jboss.brmsbpmsuite.patching.phase.PatchingPhase;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.List;

public class CommonMainArgsBuilder {
    private static Logger logger = LoggerFactory.getLogger(CommonMainArgsBuilder.class);

    private File distributionRoot;
    private DistributionType distributionType;
    private List<PatchingPhase> phases;
    private boolean printHelp;
    private int exitStatus;

    private CommonMainArgsBuilder setDistributionRoot(File distributionRoot) {
        this.distributionRoot = distributionRoot;
        return this;
    }

    private CommonMainArgsBuilder setDistributionType(DistributionType distributionType) {
        this.distributionType = distributionType;
        return this;
    }

    private CommonMainArgsBuilder setPhases(List<PatchingPhase> phases) {
        this.phases = phases;
        return this;
    }

    private void setPrintHelpAndExitStatus(boolean printHelp, int exitStatus) {
        this.printHelp = printHelp;
        this.exitStatus = exitStatus;
    }

    public CommonMainArgs build() {
        return new CommonMainArgs(distributionRoot, distributionType, phases, printHelp, exitStatus);
    }

    public CommonMainArgsBuilder parseArgs(String[] args) {
        if (args.length == 1 && "-h".equals(args[0])) {
            this.setPrintHelpAndExitStatus(true, CommonMainArgs.SUCCESSFUL_EXIT);
        }
        else if (args.length == 2 || args.length == 3) {
            if (args.length == 2) {
                logger.debug("Two arguments specified, assuming first is the path to the distribution root and second is the " +
                        "distribution type.");
            } else {
                logger.debug("Three arguments specified, assuming first is the path to the distribution root, second is the " +
                        "distribution type and third is list of phases to execute");
            }
            this.setDistributionRoot(new File(args[0]));
            try {
                this.setDistributionType(DistributionType.fromString(args[1]));
            } catch (IllegalArgumentException e) {
                this.setPrintHelpAndExitStatus(true, CommonMainArgs.UNKNOWN_DISTRIBUTION_TYPE_EXIT);
                logger.error("Unknown distribution type specified! Please select one of the supported ones.");
            }

            // third (optional) argument is the comma-separated list of patching phases to execute
            // should be used just for testing/debugging, it is not even documented
            // it needs to be in the form of "--phases=<comma-separated-list>" to prevent users accidentally specifying
            // some dummy parameter
            if (args.length == 3) {
                String argValue = args[2].trim();
                if (!argValue.startsWith("--phases=")) {
                    this.setPrintHelpAndExitStatus(true, CommonMainArgs.INVALID_ARGS_EXIT);
                    logger.error("Incorrect arguments specified! The program expects two mandatory parameters: " +
                        "<path-to-distribution> and <type-of-distribution>!");
                } else {
                    String phasesString = argValue.substring("--phases=".length());
                    // parse the comma-separated list of phases to execute
                    String[] phaseNames = phasesString.split(",");
                    List<PatchingPhase> phasesToExecute = Lists.newArrayList();
                    for (String phaseName : phaseNames) {
                        phasesToExecute.add(PatchingPhase.fromString(phaseName.trim()));
                    }
                    this.setPhases(phasesToExecute);
                }
            }
        } else {
            this.setPrintHelpAndExitStatus(true, CommonMainArgs.INVALID_ARGS_EXIT);
            logger.error("Incorrect arguments specified! The program expects two mandatory parameters: " +
                    "<path-to-distribution> and <type-of-distribution>!");
        }
        return this;
    }
}
