package org.jboss.brmsbpmsuite.patching.client;

import com.google.common.collect.Lists;
import org.jboss.brmsbpmsuite.patching.phase.PatchingPhase;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import java.io.File;

public class CommonMainArgsBuilderTest {
    private CommonMainArgsBuilder commonMainArgsBuilder;

    @Before
    public void setup() throws Exception {
        commonMainArgsBuilder = new CommonMainArgsBuilder();
    }

    @Test
    public void shouldCorrectlyParseTwoArguments() {
        CommonMainArgs commonMainArgs = commonMainArgsBuilder.parseArgs(new String[]{"some-dir", "rhpam-business-central-eap7-deployable"}).build();

        Assert.assertEquals("Unexpected arguments parsing result!", CommonMainArgs.SUCCESSFUL_EXIT, commonMainArgs.getExitStatus());
        Assert.assertEquals("Unexpected arguments parsing result!", false, commonMainArgs.shouldPrintHelp());
        Assert.assertEquals("Unexpected distribution root!", new File("some-dir"), commonMainArgs.getDistributionRoot());
        Assert.assertEquals("Unexpected distribution type!", DistributionType.RHPAM_EAP7X_BC, commonMainArgs.getDistributionType());
    }

    @Test
    public void shouldFailWhenParsingArgumentsWithUnknownDistributionType() {
        CommonMainArgs commonMainArgs = commonMainArgsBuilder.parseArgs(new String[]{"some-dir", "unknown-type"}).build();

        Assert.assertEquals("Unexpected arguments parsing result!", CommonMainArgs.UNKNOWN_DISTRIBUTION_TYPE_EXIT, commonMainArgs.getExitStatus());
        Assert.assertEquals("Unexpected arguments parsing result!", true, commonMainArgs.shouldPrintHelp());
    }

    @Test
    public void shouldParseArgumentsWithPhasesInCorrectFormat() {
        CommonMainArgs commonMainArgs = commonMainArgsBuilder.parseArgs(
                new String[]{"some-dir", "rhpam-business-central-eap7-deployable", "--phases=backup,apply"}).build();

        Assert.assertEquals("Unexpected arguments parsing result!", CommonMainArgs.SUCCESSFUL_EXIT, commonMainArgs.getExitStatus());
        Assert.assertEquals("Unexpected arguments parsing result!", false, commonMainArgs.shouldPrintHelp());
        Assert.assertEquals("Unexpected distribution root!", new File("some-dir"),  commonMainArgs.getDistributionRoot());
        Assert.assertEquals("Unexpected distribution type!", DistributionType.RHPAM_EAP7X_BC, commonMainArgs.getDistributionType());
        Assert.assertEquals("Unexpected list of phases to execute!",
                Lists.newArrayList(PatchingPhase.BACKUP, PatchingPhase.APPLY), commonMainArgs.getPhases());
        }

    @Test
    public void shouldFailWhenParsingArgumentsWithPhasesInIncorrectFormat() {
        CommonMainArgs commonMainArgs = commonMainArgsBuilder.parseArgs(
                new String[]{"some-dir", "rhpam-business-central-eap7-deployable", "incorrect-format-for-phases-list"}).build();

        Assert.assertEquals("Unexpected arguments parsing result!", CommonMainArgs.INVALID_ARGS_EXIT, commonMainArgs.getExitStatus());
        Assert.assertEquals("Unexpected arguments parsing result!", true, commonMainArgs.shouldPrintHelp());
    }

    @Test
    public void shouldFailWhenMissingArguments() {
        CommonMainArgs commonMainArgs = commonMainArgsBuilder.parseArgs(
                new String[]{"some-dir"}).build();

        Assert.assertEquals("Unexpected arguments parsing result!", CommonMainArgs.INVALID_ARGS_EXIT, commonMainArgs.getExitStatus());
        Assert.assertEquals("Unexpected arguments parsing result!", true, commonMainArgs.shouldPrintHelp());
    }
}
