package com.redhat.installer.asconfiguration.processpanel.postinstallation;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.redhat.installer.installation.processpanel.ProcessPanelHelper;
import com.redhat.installer.installation.util.InstallationUtilities;
import org.apache.commons.cli.MissingArgumentException;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.jboss.dmr.ModelNode;
import org.wildfly.security.auth.server.IdentityCredentials;
import org.wildfly.security.credential.PasswordCredential;
import org.wildfly.security.credential.store.CredentialStore;
import org.wildfly.security.credential.store.CredentialStore.CredentialSourceProtectionParameter;
import org.wildfly.security.credential.store.CredentialStore.ProtectionParameter;
import org.wildfly.security.credential.store.CredentialStoreException;
import org.wildfly.security.credential.store.WildFlyElytronCredentialStoreProvider;
import org.wildfly.security.credential.store.impl.KeyStoreCredentialStore;
import org.wildfly.security.password.Password;
import org.wildfly.security.password.WildFlyElytronPasswordProvider;
import org.wildfly.security.password.interfaces.ClearPassword;
import org.wildfly.security.util.PasswordBasedEncryptionUtil;

import java.security.GeneralSecurityException;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.Security;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class InstallCredentialStore extends PostInstallation {

    private static final String SALT = "12345678";
    private static final int ITERATION = 200;

    private static final String NAME = "name";
    private static final String PASSWORD = "password";
    private static final String LOCATION = "location";
    private static final String RELATIVE_TO = "relative-to";
    private static final String MODIFIABLE = "modifiable";
    private static final String CREATE = "create";
    private static final String KEY_STORE_TYPE = "keyStoreType";
    private static final String ALIAS = "alias";

    private static final String ARGUMENT_SPLIT_REGEX = "\\s*=\\s*";
    private static final String ALIAS_SPLIT_REGEX = "\\s*:\\s*";

    private static final WildFlyElytronPasswordProvider PASSWORD_PROVIDER = new WildFlyElytronPasswordProvider();
    private static final WildFlyElytronCredentialStoreProvider CREDENTIAL_STORE_PROVIDER = new WildFlyElytronCredentialStoreProvider();

    private final Map<String, String> parsedArguments = new HashMap<>();
    private final List<Alias> aliases = new ArrayList<>();

    private static class Alias {
        final String name;
        final String secret;

        Alias(String name, String secret) {
            this.name = name;
            this.secret = secret;
        }
    }

    @Override
    protected Class getClassName() {
        return InstallCredentialStore.class;
    }

    @Override
    protected boolean performOperation() {
        boolean result = parseArguments();
        if ("true".equals(parsedArguments.get("create"))) {
            result = result && createStore();
        }
        result = result && install();
        printResult(result);
        return result;
    }

    private String getName() {
        return parsedArguments.get(NAME);
    }

    private String getLocation() {
        return parsedArguments.get(LOCATION);
    }

    private String getRelativeTo() {
        return parsedArguments.get(RELATIVE_TO);
    }

    private String getPassword() {
        return parsedArguments.get(PASSWORD);
    }

    private boolean parseArguments() {
        for (String arg : arguments) {
            String[] split = arg.split(ARGUMENT_SPLIT_REGEX);
            String key = split[0];
            String value = split[1];
            if (key.equals(ALIAS)) {
                parseAlias(value);
            } else {
                parsedArguments.put(key, value);
            }
        }
        try {
            checkRequiredParameter(NAME);
            checkRequiredParameter(PASSWORD);
            checkRequiredParameter(LOCATION);
        } catch (MissingArgumentException e) {
            printError(e);
            return false;
        }
        return true;
    }

    private void parseAlias(String alias) {
        String[] split = alias.split(ALIAS_SPLIT_REGEX);
        aliases.add(new Alias(split[0], split[1]));
    }

    private void checkRequiredParameter(String name) throws MissingArgumentException {
        if (!parsedArguments.containsKey(name)) {
            throw new MissingArgumentException(name + " is not defined.");
        }
    }

    private boolean createStore() {
        try {
            Security.addProvider(PASSWORD_PROVIDER);
            Password storePassword = ClearPassword.createRaw(ClearPassword.ALGORITHM_CLEAR, getPassword().toCharArray());
            ProtectionParameter protectionParameter = new CredentialSourceProtectionParameter(IdentityCredentials.NONE.withCredential(new PasswordCredential(storePassword)));
            CredentialStore credentialStore = CredentialStore.getInstance(KeyStoreCredentialStore.KEY_STORE_CREDENTIAL_STORE, CREDENTIAL_STORE_PROVIDER);
            credentialStore.initialize(getStoreConfiguration(), protectionParameter, new Provider[] {PASSWORD_PROVIDER});
            addAliases(credentialStore);
            credentialStore.flush();
        } catch (NoSuchAlgorithmException | CredentialStoreException e) {
            printError(e);
            return false;
        }
        return true;
    }

    private Map<String, String> getStoreConfiguration() {
        String location = getLocation();
        if (getRelativeTo() != null && System.getProperty(getRelativeTo()) != null) {
            location = System.getProperty(getRelativeTo()) + "/" + location;
        }
        InstallationUtilities.addFileToCleanupList(location);

        Map<String, String> configuration = new HashMap<>();
        configuration.put(LOCATION, location);
        configuration.put(MODIFIABLE, "true");
        configuration.put(CREATE, "true");
        configuration.put(KEY_STORE_TYPE, "JCEKS");
        return configuration;
    }

    private void addAliases(CredentialStore credentialStore) throws CredentialStoreException {
        for (Alias alias : aliases) {
            if (!credentialStore.exists(alias.name, PasswordCredential.class)) {
                Password clearPassword = ClearPassword.createRaw(ClearPassword.ALGORITHM_CLEAR, alias.secret.toCharArray());
                credentialStore.store(alias.name, new PasswordCredential(clearPassword));
            }
        }
    }

    private boolean install() {
        List<ModelNode> results = new ArrayList<>();
        String maskedPassword;
        try {
            maskedPassword = maskPassword(getPassword());
        } catch (GeneralSecurityException e) {
            printError(e);
            return false;
        }
        results.add(serverCommands.submitCommand(installCommand(maskedPassword)));
        return installResult(results);
    }

    private String maskPassword(String password) throws GeneralSecurityException {
        PasswordBasedEncryptionUtil encryptUtil = new PasswordBasedEncryptionUtil.Builder()
                .picketBoxCompatibility()
                .salt(SALT)
                .iteration(ITERATION)
                .encryptMode()
                .build();
        return "MASK-" + encryptUtil.encryptAndEncode(password.toCharArray()) + ";" + SALT + ";" + ITERATION;
    }

    private String installCommand(String maskedPassword) {
        StringBuilder command = new StringBuilder();
        command.append("/subsystem=elytron/credential-store=").append(getName()).append(":add(location=").append(getLocation());
        if (!StringUtils.isEmpty(getRelativeTo())) {
            command.append(", relative-to=").append(getRelativeTo());
        }
        command.append(", credential-reference={clear-text=\"").append(maskedPassword).append("\"}, create=false)");
        return command.toString();
    }

    private void printError(Exception e) {
        ProcessPanelHelper.printToPanel(mHandler, e.getMessage(), true);
        ProcessPanelHelper.printToLog(ExceptionUtils.getStackTrace(e));
    }

    private void printResult(boolean result) {
        AutomatedInstallData idata = AutomatedInstallData.getInstance();
        if (result) {
            ProcessPanelHelper.printToPanel(mHandler, idata.langpack.getString("CredentialStoreInstaller.install.success"), false);
        } else {
            ProcessPanelHelper.printToPanel(mHandler, idata.langpack.getString("CredentialStoreInstaller.install.failure"), true);
        }
    }
}
