package org.jboss.brmsbpmsuite.patching.client;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.List;

/**
 * Main application class. Should be used when running the client patcher from command-line only.
 * Contains {@code System.exit()} calls which is considered harmful for standard unit tests.
 */
public class ClientPatcherApp {
    private static Logger logger = LoggerFactory.getLogger(ClientPatcherApp.class);

    protected static void commonMain(TargetProduct product, List<DistributionType> supportedDistroTypes, CommonMainArgs commonMainArgs) {
        File basedir = new File(".");
        // assume the basedir is current working directory
        ClientPatcherConfig config = new ClientPatcherConfig();
        config.setProduct(product);
        config.setSupportedDistroTypes(supportedDistroTypes);
        config.setBackupBaseDir(new File(basedir, "backup"));
        config.setDistributionRoot(commonMainArgs.getDistributionRoot());
        config.setDistributionType(commonMainArgs.getDistributionType());
        config.setPhasesToExecute(commonMainArgs.getPhases());

        try {
            // Recognize the distribution
            DistributionRecognizer distributionRecognizer = new DistributionRecognizerImpl();
            config.setDistributionExploded(distributionRecognizer.isDistributionExploded(config.getDistributionRoot()));

            new ClientPatcherRunner(config).run();
        } catch (ClientPatcherException e) {
            logger.error(e.getMessage());
            printHelp(supportedDistroTypes);
            System.exit(2);
        } catch (Exception e) {
            logger.error("Unexpected error occurred. Please contact Red Hat Support if you think this is a bug in the " +
                    "updating tool itself.", e);
            System.exit(10);
        }
    }

    protected static void printHelp(List<DistributionType> supportedDistroTypes) {
        StringBuilder helpSB = new StringBuilder();
        helpSB.append("\nUsage: apply-updates.[sh|bat] <path-to-distribution-root> <type-of-distribution>\n");
        helpSB.append("\nDescription: The Red Hat Decision Manager and Red Hat Process Automation Manager client update tool used to apply patch updates and minor release upgrades to existing installations.");

        helpSB.append("\n\nIMPORTANT: Do not apply the updates while the applications are running. Shut down KIE Server first.");

        helpSB.append("\n\nSupported distribution types:\n");
        for (DistributionType distroType : supportedDistroTypes) {
            helpSB.append("\t- " + distroType.getName() + ": " + distroType.getDescription() +"\n");
        }
        
        helpSB.append("\n\nExamples:\n");
        helpSB.append("\n\t- Update to Business Central and KIE Server for a full Red Hat Process Automation Manager distribution on Red Hat JBoss EAP:\n" +
                        "\n\t\t./apply-updates.sh ~EAP_HOME/standalone/deployments/business-central.war rhpam-business-central-eap7-deployable\n\n" +
                        "\t\t./apply-updates.sh ~EAP_HOME/standalone/deployments/kie-server.war rhpam-kie-server-ee8\n");
        helpSB.append("\n\t- Update to headless Red Hat Process Automation Manager controller, if used:\n" +
                "\n\t\t./apply-updates.sh ~EAP_HOME/standalone/deployments/controller.war rhpam-controller-ee7\n");

        helpSB.append("\nNotes:\n" +
        "\t- Working dir needs to be the directory of this script!\n" +
        "\t- Java is recommended to be JDK and version 8 or later\n" +
        "\t- The environment variable JAVA_HOME should be set to the JDK installation directory\n" +
        "\t\tExample for Linux: export JAVA_HOME=/usr/lib/jvm/java-1.8.0\n" +
        "\t\tExample for Mac: export JAVA_HOME=/Library/Java/Home"
        );

        System.out.println(helpSB.toString());
    }

}
