package com.redhat.installer.asconfiguration.jdbc.validator;

import com.izforge.izpack.LocaleDatabase;
import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.DataValidator;
import com.redhat.installer.asconfiguration.jdbc.constant.JBossJDBCConstants;
import com.redhat.installer.asconfiguration.utils.JarUtils;

import java.util.HashSet;
import java.util.Set;

/**
 * Created by aabulawi on 30/04/15.
 */
public class VerifyJDBCJarsValidator implements DataValidator{

    private String formatedMessage;

    @Override
    public Status validateData(AutomatedInstallData adata) {
        Set<String> jarPathSet = new HashSet<String>();
        for (String key : adata.getVariables().stringPropertyNames()){
            if (key.matches("^jdbc.driver.jar-[0-9]+-path$")){
                adata.getVariable(key);
                jarPathSet.add(adata.getVariable(key));
            }
        }

        if (jarPathSet.isEmpty()){
            setFormattedMessage(String.format(adata.langpack.getString("JBossJDBCDriverSetupPanel.path.nopaths")));
            return Status.ERROR;
        }

        for(String jarPath : jarPathSet){
            if (!validJarPath(jarPath, adata.langpack)){
                return Status.ERROR;
            }
        }

        String driverName = adata.getVariable("jdbc.driver.vendor.name");
        Object[] jarPaths = jarPathSet.toArray();
        if(!validJDBCDriver(driverName, jarPaths)){
            setFormattedMessage(String.format(adata.langpack.getString("JBossJDBCDriverSetupPanel.path.nodriver"), getJDBCClassName(driverName)));
            return Status.WARNING;
        }

        return Status.OK;
    }

    @Override
    public String getErrorMessageId() {
        return null;
    }

    @Override
    public String getWarningMessageId() {
        return null;
    }

    @Override
    public boolean getDefaultAnswer() {
        return false;
    }

    @Override
    public String getFormattedMessage() {
        return formatedMessage;
    }

    public void setFormattedMessage(String message) {
        formatedMessage = message;
    }

    private boolean validJarPath(String jarPath, LocaleDatabase langpack) {
        String safeDisplayPath;
        if (jarPath.length() > 95) { // the length which causes the dialog to render poorly
            int startIndex = jarPath.length() - 80;
            safeDisplayPath = "..."+jarPath.substring(startIndex);
        } else {
            safeDisplayPath = jarPath;
        }
        switch(JarUtils.verifyJarPath(jarPath)){
            case 0: // jar is fine
                break;
            case 1: // given path doesn't exist (or is directory)
                setFormattedMessage(String.format(langpack.getString("JBossJDBCDriverSetupPanel.path.error"), safeDisplayPath));
                return false;
            case 2: // given path isn't a zip
                setFormattedMessage(String.format(langpack.getString("JBossJDBCDriverSetupPanel.path.notzip"), safeDisplayPath));
                return false;
            case 3: // given path is a zip, but an empty one
                setFormattedMessage(String.format(langpack.getString("JBossJDBCDriverSetupPanel.path.emptyzip"), safeDisplayPath));
                return false;
            case 4: // given remote location isn't reachable
                setFormattedMessage(String.format(langpack.getString("JBossJDBCDriverSetupPanel.path.remote.error"), safeDisplayPath));
                return false;
            case 5:
                setFormattedMessage(String.format(langpack.getString("JBossJDBCDriverSetupPanel.path.exception.error"), safeDisplayPath));
                return false;
        }
        return true;
    }

    private boolean validJDBCDriver(String jdbcName, Object[] jarPaths) {

        Class<?> driverClass = JDBCConnectionUtils.findDriverClass(JBossJDBCConstants.CLASSNAME_MAP.get(jdbcName), JDBCConnectionUtils.convertToUrlArray(jarPaths));
        if (driverClass == null) {
            return false;
        }
        return true;
    }

    private String getJDBCClassName(String jdbcName){
        if (jdbcName.equals(JBossJDBCConstants.IBM_JDBC_NAME))
            return JBossJDBCConstants.IBM_DRIVER_CLASS_CHECK;
        else if (jdbcName.equals(JBossJDBCConstants.MYSQL_JDBC_NAME))
            return JBossJDBCConstants.MYSQL_DRIVER_CLASS_CHECK;
        else if (jdbcName.equals(JBossJDBCConstants.SYBASE_JDBC_NAME))
            return JBossJDBCConstants.SYBASE_DRIVER_CLASS_CHECK;
        else if (jdbcName.equals(JBossJDBCConstants.POSTGRESQL_JDBC_NAME))
            return JBossJDBCConstants.POSTGRESQL_DRIVER_CLASS_CHECK;
        else if (jdbcName.equals(JBossJDBCConstants.MICROSOFT_JDBC_NAME))
            return JBossJDBCConstants.MICROSOFT_DRIVER_CLASS_CHECK;
        else if (jdbcName.equals(JBossJDBCConstants.ORACLE_JDBC_NAME))
            return JBossJDBCConstants.ORACLE_DRIVER_CLASS_CHECK;
        else if (jdbcName.equals(JBossJDBCConstants.MARIADB_JDBC_NAME))
            return JBossJDBCConstants.MARIADB_DRIVER_CLASS_CHECK;

        return null;
    }
}
