/*
 * Copyright 2020 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.kie.workbench.common.stunner.client.widgets.editor;

import java.util.function.Consumer;

import javax.annotation.PreDestroy;
import javax.enterprise.context.Dependent;
import javax.enterprise.inject.Any;
import javax.inject.Inject;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.user.client.ui.IsWidget;
import org.jboss.errai.ioc.client.api.ManagedInstance;
import org.kie.workbench.common.stunner.client.widgets.presenters.session.SessionDiagramPresenter;
import org.kie.workbench.common.stunner.client.widgets.presenters.session.SessionPresenter;
import org.kie.workbench.common.stunner.client.widgets.presenters.session.impl.SessionEditorPresenter;
import org.kie.workbench.common.stunner.client.widgets.presenters.session.impl.SessionViewerPresenter;
import org.kie.workbench.common.stunner.core.client.canvas.CanvasHandler;
import org.kie.workbench.common.stunner.core.client.i18n.ClientTranslationService;
import org.kie.workbench.common.stunner.core.client.service.ClientRuntimeError;
import org.kie.workbench.common.stunner.core.client.session.ClientSession;
import org.kie.workbench.common.stunner.core.client.session.impl.EditorSession;
import org.kie.workbench.common.stunner.core.client.session.impl.ViewerSession;
import org.kie.workbench.common.stunner.core.definition.exception.DefinitionNotFoundException;
import org.kie.workbench.common.stunner.core.diagram.Diagram;
import org.kie.workbench.common.stunner.core.diagram.DiagramParsingException;
import org.kie.workbench.common.stunner.core.i18n.CoreTranslationMessages;
import org.uberfire.client.workbench.widgets.common.ErrorPopupPresenter;
import org.uberfire.ext.widgets.common.client.ace.AceEditorMode;
import org.uberfire.ext.widgets.core.client.editors.texteditor.TextEditorView;

@Dependent
public class StunnerEditor {

    private final ManagedInstance<SessionEditorPresenter<EditorSession>> editorSessionPresenterInstances;
    private final ManagedInstance<SessionViewerPresenter<ViewerSession>> viewerSessionPresenterInstances;
    private final ClientTranslationService translationService;
    private final ManagedInstance<TextEditorView> xmlEditorViews;
    private final ErrorPopupPresenter errorPopupPresenter;
    private final StunnerEditorView view;

    private SessionDiagramPresenter diagramPresenter;
    private TextEditorView xmlEditorView;
    private boolean isReadOnly;
    private Consumer<DiagramParsingException> parsingExceptionProcessor;
    private Consumer<Throwable> exceptionProcessor;
    private Consumer<Integer> onResetContentHashProcessor;
    private int contentHash;

    // CDI proxy.
    public StunnerEditor() {
        this(null, null, null, null, null, null);
    }

    @Inject
    public StunnerEditor(ManagedInstance<SessionEditorPresenter<EditorSession>> editorSessionPresenterInstances,
                         ManagedInstance<SessionViewerPresenter<ViewerSession>> viewerSessionPresenterInstances,
                         ClientTranslationService translationService,
                         @Any ManagedInstance<TextEditorView> xmlEditorViews,
                         ErrorPopupPresenter errorPopupPresenter,
                         StunnerEditorView view) {
        this.editorSessionPresenterInstances = editorSessionPresenterInstances;
        this.viewerSessionPresenterInstances = viewerSessionPresenterInstances;
        this.translationService = translationService;
        this.xmlEditorViews = xmlEditorViews;
        this.errorPopupPresenter = errorPopupPresenter;
        this.isReadOnly = false;
        this.view = view;
        this.parsingExceptionProcessor = e -> {
        };
        this.exceptionProcessor = e -> {
        };
        this.onResetContentHashProcessor = e -> {
        };
    }

    public void setReadOnly(boolean readOnly) {
        isReadOnly = readOnly;
    }

    public void setOnResetContentHashProcessor(Consumer<Integer> onResetContentHashProcessor) {
        this.onResetContentHashProcessor = onResetContentHashProcessor;
    }

    public void setParsingExceptionProcessor(Consumer<DiagramParsingException> parsingExceptionProcessor) {
        this.parsingExceptionProcessor = parsingExceptionProcessor;
    }

    public void setExceptionProcessor(Consumer<Throwable> exceptionProcessor) {
        this.exceptionProcessor = exceptionProcessor;
    }

    public void open(final Diagram diagram,
                     final SessionPresenter.SessionPresenterCallback callback) {
        if (isClosed()) {
            if (!isReadOnly) {
                diagramPresenter = editorSessionPresenterInstances.get();
            } else {
                diagramPresenter = viewerSessionPresenterInstances.get();
            }
            diagramPresenter.displayNotifications(type -> true);
            diagramPresenter.withPalette(!isReadOnly);
            diagramPresenter.withToolbar(false);
            view.setWidget(diagramPresenter.getView());
        }
        diagramPresenter.open(diagram, new SessionPresenter.SessionPresenterCallback() {
            @Override
            public void onOpen(Diagram diagram) {
                callback.onOpen(diagram);
            }

            @Override
            public void afterSessionOpened() {
                callback.afterSessionOpened();
            }

            @Override
            public void afterCanvasInitialized() {
                callback.afterCanvasInitialized();
            }

            @Override
            public void onSuccess() {
                callback.onSuccess();
                resetContentHash();
            }

            @Override
            public void onError(ClientRuntimeError error) {
                handleError(error);
                callback.onError(error);
            }
        });
    }

    public int getCurrentContentHash() {
        if (isXmlEditorEnabled()) {
            return xmlEditorView.getContent().hashCode();
        }
        if (null == getSession()) {
            return 0;
        }
        if (null == getCanvasHandler().getDiagram()) {
            return 0;
        }
        return getCanvasHandler().getDiagram().hashCode();
    }

    public void resetContentHash() {
        contentHash = getCurrentContentHash();
        onResetContentHashProcessor.accept(contentHash);
    }

    public boolean isDirty() {
        return contentHash != getCurrentContentHash();
    }

    public void displayXML(String xml) {
        displayXMLFlat(xml);
    }

    protected void displayXMLFlat(String xml) {
        close();
        if (xmlEditorViews == null) {
            return;
        }
        xmlEditorView = xmlEditorViews.get();
        xmlEditorView.setReadOnly(isReadOnly());
        xmlEditorView.setContent(xml, AceEditorMode.XML);
        resetContentHash();
        view.setWidget(xmlEditorView.asWidget());
        Scheduler.get().scheduleDeferred(xmlEditorView::onResize);
    }

    public void handleError(final ClientRuntimeError error) {
        final Throwable e = error.getThrowable();
        if (e instanceof DiagramParsingException) {
            final DiagramParsingException dpe = (DiagramParsingException) e;
            parsingExceptionProcessor.accept(dpe);
            displayXMLFlat(dpe.getXml());
        } else {
            String message = null;
            if (e instanceof DefinitionNotFoundException) {
                final DefinitionNotFoundException dnfe = (DefinitionNotFoundException) e;
                message = translationService.getValue(CoreTranslationMessages.DIAGRAM_LOAD_FAIL_UNSUPPORTED_ELEMENTS,
                                                      dnfe.getDefinitionId());
            } else {
                message = error.getThrowable() != null ?
                        error.getThrowable().getMessage() : error.getMessage();
            }
            showError(message);
            exceptionProcessor.accept(error.getThrowable());
        }
    }

    public void focus() {
        if (!isClosed()) {
            diagramPresenter.focus();
        }
    }

    public void lostFocus() {
        if (!isClosed()) {
            diagramPresenter.lostFocus();
        }
    }

    public StunnerEditor close() {
        if (!isClosed()) {
            diagramPresenter.destroy();
            diagramPresenter = null;
            editorSessionPresenterInstances.destroyAll();
            viewerSessionPresenterInstances.destroyAll();
            view.clear();
        }
        if (null != xmlEditorView) {
            xmlEditorViews.destroyAll();
            xmlEditorView = null;
        }
        return this;
    }

    @PreDestroy
    public void destroy() {
        close();
    }

    public boolean isReadOnly() {
        return isReadOnly;
    }

    public boolean isClosed() {
        return null == diagramPresenter;
    }

    public ClientSession getSession() {
        return (ClientSession) diagramPresenter.getInstance();
    }

    public CanvasHandler getCanvasHandler() {
        return (CanvasHandler) diagramPresenter.getHandler();
    }

    public Diagram getDiagram() {
        return getCanvasHandler().getDiagram();
    }

    public SessionDiagramPresenter getPresenter() {
        return diagramPresenter;
    }

    public boolean isXmlEditorEnabled() {
        return null != xmlEditorView;
    }

    public void showMessage(String message) {
        diagramPresenter.getView().showMessage(message);
    }

    public void showError(String message) {
        errorPopupPresenter.showMessage(message);
    }

    public IsWidget getView() {
        return view;
    }

    public TextEditorView getXmlEditorView() {
        return xmlEditorView;
    }
}
