/*
 * Copyright 2019 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.kie.workbench.common.dmn.client.editors.search;

import java.util.List;
import java.util.function.Supplier;

import javax.annotation.PostConstruct;
import javax.enterprise.context.Dependent;
import javax.inject.Inject;

import org.kie.workbench.common.dmn.client.editors.expressions.ExpressionEditor;
import org.kie.workbench.common.dmn.client.session.DMNSession;
import org.kie.workbench.common.stunner.core.client.api.SessionManager;
import org.kie.workbench.common.widgets.client.search.common.BaseEditorSearchIndex;
import org.uberfire.mvp.Command;

import static org.kie.workbench.common.dmn.client.editors.search.DMNEditorSearchIndex.SearchContext.BOXED_EXPRESSION;
import static org.kie.workbench.common.dmn.client.editors.search.DMNEditorSearchIndex.SearchContext.DATA_TYPES;
import static org.kie.workbench.common.dmn.client.editors.search.DMNEditorSearchIndex.SearchContext.GRAPH;

@Dependent
public class DMNEditorSearchIndex extends BaseEditorSearchIndex<DMNSearchableElement> {

    private final DMNGraphSubIndex graphSubIndex;

    private final DMNGridSubIndex gridSubIndex;

    private final DMNDataTypesSubIndex dataTypesSubIndex;

    private final SessionManager sessionManager;

    private Supplier<Boolean> isDataTypesTabActiveSupplier = () -> false;

    @Inject
    public DMNEditorSearchIndex(final DMNGraphSubIndex graphSubIndex,
                                final DMNGridSubIndex gridSubIndex,
                                final DMNDataTypesSubIndex dataTypesSubIndex,
                                final SessionManager sessionManager) {
        this.graphSubIndex = graphSubIndex;
        this.gridSubIndex = gridSubIndex;
        this.dataTypesSubIndex = dataTypesSubIndex;
        this.sessionManager = sessionManager;
    }

    @PostConstruct
    public void init() {
        registerSubIndex(graphSubIndex);
        registerSubIndex(gridSubIndex);
        setNoResultsFoundCallback(getNoResultsFoundCallback());
        setSearchClosedCallback(getSearchClosedCallback());
    }

    @Override
    protected List<DMNSearchableElement> getSearchableElements() {
        return getSubIndex().getSearchableElements();
    }

    Command getNoResultsFoundCallback() {
        return () -> getSubIndex().onNoResultsFound();
    }

    public void setIsDataTypesTabActiveSupplier(final Supplier<Boolean> isDataTypesTabActiveSupplier) {
        this.isDataTypesTabActiveSupplier = isDataTypesTabActiveSupplier;
    }

    Command getSearchClosedCallback() {
        return () -> getSubIndex().onSearchClosed();
    }

    private DMNSubIndex getSubIndex() {
        final SearchContext searchContext = currentSearchContext();
        switch (searchContext) {
            case BOXED_EXPRESSION:
                return gridSubIndex;
            case DATA_TYPES:
                return dataTypesSubIndex;
            case GRAPH:
                return graphSubIndex;
            default:
                throw new UnsupportedOperationException("Unsupported search context: " + searchContext);
        }
    }

    public Supplier<Boolean> getIsDataTypesTabActiveSupplier() {
        return isDataTypesTabActiveSupplier;
    }

    private SearchContext currentSearchContext() {
        if (getIsDataTypesTabActiveSupplier().get()) {
            return DATA_TYPES;
        }
        if (getExpressionEditor().isActive()) {
            return BOXED_EXPRESSION;
        }
        return GRAPH;
    }

    private ExpressionEditor getExpressionEditor() {
        return (ExpressionEditor) getCurrentSession().getExpressionEditor();
    }

    private DMNSession getCurrentSession() {
        return sessionManager.getCurrentSession();
    }

    enum SearchContext {
        BOXED_EXPRESSION,
        DATA_TYPES,
        GRAPH
    }
}
