package org.jboss.brmsbpmsuite.patching.client;

import com.google.common.base.Strings;
import org.jboss.brmsbpmsuite.patching.phase.BasePatchingPhaseRunner;
import org.jboss.brmsbpmsuite.patching.phase.DistributionPatchingPhaseRunner;
import org.jboss.brmsbpmsuite.patching.phase.PatchingPhase;
import org.jboss.brmsbpmsuite.patching.phase.PatchingPhaseRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import static org.jboss.brmsbpmsuite.patching.phase.PatchingPhase.*;

/**
 * Responsible for running the actual client patcher and its phases.
 */
public class ClientPatcherRunner {
    private static final Logger logger = LoggerFactory.getLogger(ClientPatcherRunner.class);

    final ClientPatcherConfig config;

    public ClientPatcherRunner(ClientPatcherConfig config) {
        this.config = config;
    }

    public void run() {
        config.validate();
        logger.debug("Running client patcher with following configuration: {}", config);
        logger.debug("Executing following phases: " + config.getPhasesToExecute());

        DistributionPatcher distributionPatcher = PatcherFactory.newDistributionPatcher(config);
        runDistributionPatcher(distributionPatcher, backupDir(""));

        Patcher configPatcher = PatcherFactory.newConfigPatcher(config);
        runConfigPatcher(configPatcher, backupDir("configuration"));
    }

    private File backupDir(String childDirName) {
        String backupChildDir = config.getDistributionType().getName();
        if (!Strings.isNullOrEmpty(childDirName)) {
            backupChildDir += "-" + childDirName;
        }
        return new File(config.getBackupDir(), backupChildDir);
    }

    public void runDistributionPatcher(DistributionPatcher patcher, File backupDir) {
        List<PatchingPhase> patchingSequence = new ArrayList<PatchingPhase>() {{
            add(EXPLODE_WAR); add(CHECK_DISTRO); add(CHECK_VERSION); add(APPLY); add(ZIP_WAR);
        }};

        int backupPosition = config.isDistributionExploded() ? patchingSequence.indexOf(APPLY) : 0;
        patchingSequence.add(backupPosition, BACKUP);

        PatchingPhaseRunner phaseRunner = new DistributionPatchingPhaseRunner(patcher, backupDir);
        runPhases(phaseRunner, "distribution", patchingSequence);
    }

    public void runConfigPatcher(Patcher patcher, File backupDir) {
        PatchingPhaseRunner phaseRunner = new BasePatchingPhaseRunner(patcher, backupDir);
        runPhases(phaseRunner, "configuration", Arrays.asList(BACKUP, APPLY));
    }

    private void runPhases(PatchingPhaseRunner phaseRunner, String qualifier, List<PatchingPhase> patchingSequence) {
        List<PatchingPhase> phasesToExecute = filterPhases(patchingSequence);
        for (PatchingPhase phase : phasesToExecute) {
            try {
                phaseRunner.run(phase);
            } catch (IOException e) {
                String msg = "Unexpected error occurred when " + phase.action + " the " + qualifier + "!";
                logger.error(msg, e);
                throw new ClientPatcherException(msg, e);
            }
        }
    }

    private List<PatchingPhase> filterPhases(List<PatchingPhase> patchingSequence) {
        return patchingSequence.stream().
                filter(phase -> config.getPhasesToExecute().contains(phase)).
                collect(Collectors.toList());
    }

}
