package org.jboss.brmsbpmsuite.patching.client;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.*;

public class PatcherFactory {
    private static final Logger logger = LoggerFactory.getLogger(PatcherFactory.class);

    private static final String BLACKLIST_FILENAME = "blacklist.txt";
    private static final String REMOVE_LIST_FILENAME = "remove-list.txt";
    private static final String CHECKSUMS_FILENAME = "checksums.txt";
    private static final String SHARED_NEW_CONTENT_DIRNAME = "new-content";
    private static final String NEW_CONTENT_FILENAME = "new-content.txt";

    private static final Patcher NOOP_PATCHER = new Patcher() {
        @Override
        public void backup(File backupDir) {}

        @Override
        public void apply() {}
    };

    public static DistributionPatcher newDistributionPatcher(ClientPatcherConfig config) {
        DistributionType distributionType = config.getDistributionType();
        DistributionChecker distroChecker = DistributionCheckerFactory.create(distributionType, config.getProduct());
        VersionChecker versionChecker = VersionCheckerProvider.getInstance();
        Patcher directoryPatcher = newDirectoryPatcher(config.getDistributionRoot(), config.getPatchBasedir(),
                distributionType.getRelativePath());
        return new GeneralDistributionPatcher(config.getDistributionRoot(), distroChecker, versionChecker, directoryPatcher);
    }

    public static DirectoryPatcher newDirectoryPatcher(File rootDir, File patchDir, String patchName) {
        File distroUpdatesDir = new File(patchDir, "updates/" + patchName);
        File sharedNewContentDir = new File(patchDir, "updates/" + SHARED_NEW_CONTENT_DIRNAME);
        File distroNewContentFile = new File(distroUpdatesDir, NEW_CONTENT_FILENAME);
        File blacklist = new File(patchDir, BLACKLIST_FILENAME);

        return new DirectoryPatcher(rootDir,
                readLines(new File(distroUpdatesDir, REMOVE_LIST_FILENAME)),
                getBlacklistedPaths(blacklist),
                createPatchEntries(distroNewContentFile, sharedNewContentDir),
                getChecksums(new File(distroUpdatesDir, CHECKSUMS_FILENAME)));
    }

    private static Map<String, List<Checksum>> getChecksums(File checksumsFile) {
        logger.debug("Parsing checksums file {}", checksumsFile.getAbsolutePath());
        Properties checksumProps;
        try {
            checksumProps = new Properties();
            checksumProps.load(new FileReader(checksumsFile));
        } catch (IOException e) {
            String msg = "Can not read checksums inside properties file " + checksumsFile.getAbsolutePath();
            logger.error(msg);
            throw new RuntimeException(msg, e);
        }
        Map<String, List<Checksum>> checksumsMap = new HashMap<String, List<Checksum>>();
        for (String path: checksumProps.stringPropertyNames()) {
            logger.trace("Parsing checksums for path '{}'", path);
            String checksumsStr = checksumProps.getProperty(path);
            if (checksumsStr == null || checksumsStr.equals("")) {
                throw new RuntimeException("No checksums provided for path " + path + " (and the path is specific in the file). "
                 + "Path can be specified only if there is at one checksum associated with it.");
            }
            checksumsMap.put(path, parseChecksums(checksumsStr));

        }
        return checksumsMap;
    }

    private static List<Checksum> parseChecksums(String checksumsStr) {
        List<Checksum> checksums = new ArrayList<Checksum>();
        String[] parts = checksumsStr.trim().split(",");
        for (String checksum : parts) {
            checksums.add(Checksum.md5(checksum));
        }
        return checksums;
    }

    private static List<String> getBlacklistedPaths(File blacklist) {
        if (blacklist.exists() && blacklist.isFile()) {
            logger.info("File blacklist.txt found at {}.", blacklist.getAbsolutePath());
            return readLines(blacklist);
        } else {
            logger.info("File blacklist.txt _not_ found. It was expected at {}.", blacklist.getAbsolutePath());
            return Collections.emptyList();
        }
    }

    private static List<String> readLines(File file) {
        List<String> allLines;
        try {
            allLines = FileUtils.readLines(file);
        } catch (IOException e) {
            String msg = "Can not read content of file " + file.getAbsolutePath();
            logger.error(msg, e);
            throw new ClientPatcherException(msg, e);
        }
        // remove empty lines and comments (lines starting with #)
        List<String> filtered = new ArrayList<String>();
        for (String line : allLines) {
            if (line.trim().isEmpty() || line.trim().startsWith("#")) {
                continue;
            }
            filtered.add(line);
        }
        return filtered;
    }

    private static List<PatchEntry> createPatchEntries(File newContentFile, File newContentSharedDir) {
        List<PatchEntry> patchEntries = new ArrayList<PatchEntry>();
        for (String line : readLines(newContentFile)) {
            String[] parts = line.split("=");
            String relativePath = parts[0];
            String newContentFilename = parts[1];
            patchEntries.add(new PatchEntry(relativePath, new File(newContentSharedDir, newContentFilename)));
        }
        logger.trace("Created patch entries:" + patchEntries);
        return patchEntries;
    }

    public static Patcher newConfigPatcher(ClientPatcherConfig config) {
        if (config.getDistributionType() != DistributionType.RHPAM_EAP7X_BC
                && config.getDistributionType() != DistributionType.RHDM_EAP7X_DC) {
            return NOOP_PATCHER;
        }

        File patchesDir = new File(config.getPatchBasedir(), "updates/configuration");
        File standaloneDir = config.getDistributionRoot().getParentFile().getParentFile();
        File configDir = new File(standaloneDir, "configuration");
        try {
            return new ConfigurationPatcher(patchesDir, configDir);
        } catch (ParserConfigurationException | TransformerConfigurationException e) {
            throw new IllegalStateException("Could not instantiate the configuration patcher.", e);
        }
    }

}
