/*
 * Copyright 2020 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kie.workbench.common.stunner.bpmn.definition;

import javax.validation.Valid;

import org.jboss.errai.common.client.api.annotations.MapsTo;
import org.jboss.errai.common.client.api.annotations.Portable;
import org.jboss.errai.databinding.client.api.Bindable;
import org.kie.workbench.common.forms.adf.definitions.annotations.FieldParam;
import org.kie.workbench.common.forms.adf.definitions.annotations.FormDefinition;
import org.kie.workbench.common.forms.adf.definitions.annotations.FormField;
import org.kie.workbench.common.forms.adf.definitions.settings.FieldPolicy;
import org.kie.workbench.common.stunner.bpmn.definition.property.background.BackgroundSet;
import org.kie.workbench.common.stunner.bpmn.definition.property.dimensions.CircleDimensionSet;
import org.kie.workbench.common.stunner.bpmn.definition.property.dimensions.Radius;
import org.kie.workbench.common.stunner.bpmn.definition.property.event.link.LinkEventExecutionSet;
import org.kie.workbench.common.stunner.bpmn.definition.property.font.FontSet;
import org.kie.workbench.common.stunner.bpmn.definition.property.general.BPMNGeneralSet;
import org.kie.workbench.common.stunner.core.definition.annotation.Definition;
import org.kie.workbench.common.stunner.core.definition.annotation.Property;
import org.kie.workbench.common.stunner.core.definition.annotation.morph.Morph;
import org.kie.workbench.common.stunner.core.util.HashUtil;

import static org.kie.workbench.common.forms.adf.engine.shared.formGeneration.processing.fields.fieldInitializers.nestedForms.AbstractEmbeddedFormsInitializer.COLLAPSIBLE_CONTAINER;
import static org.kie.workbench.common.forms.adf.engine.shared.formGeneration.processing.fields.fieldInitializers.nestedForms.AbstractEmbeddedFormsInitializer.FIELD_CONTAINER_PARAM;

@Portable
@Bindable
@Definition
@Morph(base = BaseCatchingIntermediateEvent.class)
@FormDefinition(
        startElement = "general",
        policy = FieldPolicy.ONLY_MARKED,
        defaultFieldSettings = {@FieldParam(name = FIELD_CONTAINER_PARAM, value = COLLAPSIBLE_CONTAINER)}
)
public class IntermediateLinkEventCatching extends BaseCatchingIntermediateEvent {

    @Property
    @FormField(afterElement = "general")
    @Valid
    protected LinkEventExecutionSet executionSet;

    public IntermediateLinkEventCatching() {
        this(new BPMNGeneralSet(""),
             new LinkEventExecutionSet(),
             new BackgroundSet(),
             new FontSet(),
             new CircleDimensionSet(new Radius()));
    }

    public IntermediateLinkEventCatching(final @MapsTo("general") BPMNGeneralSet general,
                                         final @MapsTo("executionSet") LinkEventExecutionSet executionSet,
                                         final @MapsTo("backgroundSet") BackgroundSet backgroundSet,
                                         final @MapsTo("fontSet") FontSet fontSet,
                                         final @MapsTo("dimensionsSet") CircleDimensionSet dimensionsSet) {
        super(general,
              backgroundSet,
              fontSet,
              dimensionsSet);
        this.executionSet = executionSet;
    }

    @Override
    protected void initLabels() {
        super.initLabels();
        // Link Events can't be boundary events
        labels.remove("IntermediateEventOnSubprocessBoundary");
        labels.remove("IntermediateEventOnActivityBoundary");
        labels.remove("EventOnChoreographyActivityBoundary");
        // Link Catch Event can't have incoming connection
        labels.add("Startevents_all");
    }

    public LinkEventExecutionSet getExecutionSet() {
        return executionSet;
    }

    public void setExecutionSet(LinkEventExecutionSet executionSet) {
        this.executionSet = executionSet;
    }

    @Override
    public int hashCode() {
        return HashUtil.combineHashCodes(super.hashCode(),
                                         executionSet.hashCode());
    }

    @Override
    public boolean equals(Object o) {
        if (o instanceof IntermediateLinkEventCatching) {
            IntermediateLinkEventCatching other = (IntermediateLinkEventCatching) o;
            return super.equals(other)
                    && executionSet.equals(other.executionSet);
        }
        return false;
    }
}
