package main

import (
	"flag"
	"fmt"
	"go/ast"
	"go/parser"
	"go/token"
	"log"
	"os"
	"regexp"
	"strings"
	"text/template"
)

const tmpl = `
// Code generated by network-observer codegen. DO NOT EDIT
package {{ .PackageName }}

// Implements ResponseSetter and CollectionResponseSetter for the generated
// response objects

{{range .ResponseTypes}}{{$typeName := .TypeName}}
{{- range .Fields}}
// Set{{ .FieldName }}
func (r *{{ $typeName }}) Set{{ .FieldName }}(v {{ .FieldType }}) {
  r.{{ .FieldName }} = v
}
{{- end}}{{end}}

// Implements Record interface for the generated record objects
{{range .RecordTypes}}{{$typeName := .TypeName}}
{{- range .Fields}}
// Get{{ .FieldName }}
func (r {{ $typeName }}) Get{{ .FieldName }}() {{ .FieldType }} {
  return r.{{ .FieldName }}
}
{{- end}}{{end}}
`

var (
	responseFieldTagRegexp = regexp.MustCompile(`json:"(results|count|timeRangeCount)[,"]`)
	recordFieldTagRegexp   = regexp.MustCompile(`json:"(startTime|endTime)[,"]`)
)

type Codegen struct {
	PackageName   string
	ResponseTypes []StructType
	RecordTypes   []StructType
}

type StructType struct {
	TypeName string
	Fields   []FieldInfo
}

type FieldInfo struct {
	FieldName string
	FieldType string
}

func main() {
	var output string
	flags := flag.NewFlagSet(os.Args[0], flag.ExitOnError)
	flags.StringVar(&output, "o", "", "file to write output to. Defaults to stdout")
	flags.Usage = func() {
		fmt.Println("codegen [options...] <source file>")
		fmt.Println(`generates getters and setters on openapi generated types.`)
		flags.PrintDefaults()
	}
	flags.Parse(os.Args[1:])

	input := flags.Args()
	if len(input) != 1 {
		flags.Usage()
		os.Exit(1)
	}
	target := input[0]
	if _, err := os.Stat(target); err != nil {
		log.Fatalf("error opening input file: %s", err)
	}

	fset := token.NewFileSet()
	node, err := parser.ParseFile(fset, target, nil, parser.ParseComments)
	if err != nil {
		log.Fatalf("could not parse go file: %s", err)
	}

	codegen := Codegen{
		PackageName: "api",
	}
	// walk ast for struct fields with json tag
	ast.Inspect(node, func(n ast.Node) bool {
		ts, ok := n.(*ast.TypeSpec)
		if !ok {
			return true
		}
		st, ok := ts.Type.(*ast.StructType)
		if !ok {
			return false
		}
		switch {
		case strings.HasSuffix(ts.Name.String(), "Response"):
			setters := fieldsMatchingType(st, responseFieldTagRegexp)
			if len(setters) > 0 {
				codegen.ResponseTypes = append(codegen.ResponseTypes, StructType{
					TypeName: ts.Name.String(),
					Fields:   setters,
				})
			}
		case strings.HasSuffix(ts.Name.String(), "Record"):
			getters := fieldsMatchingType(st, recordFieldTagRegexp)
			if len(getters) > 0 {
				codegen.RecordTypes = append(codegen.RecordTypes, StructType{
					TypeName: ts.Name.String(),
					Fields:   getters,
				})
			}
		default:
			//return false
		}
		return true
	})
	t := template.Must(template.New("codegen").Parse(tmpl))

	f := os.Stdout
	if output != "" {
		of, err := os.Create(output)
		if err != nil {
			flags.Usage()
			log.Fatalf("could not open output file: %s", err)
		}
		f = of
	}
	t.Execute(f, codegen)
}

func fieldsMatchingType(st *ast.StructType, re *regexp.Regexp) []FieldInfo {
	var fields []FieldInfo
	for _, field := range st.Fields.List {
		if len(field.Names) != 1 {
			continue
		}
		if field.Tag != nil {
			if re.MatchString(field.Tag.Value) {
				var rt string

				for typ := field.Type; typ != nil; {
					switch f := typ.(type) {
					case *ast.Ident:
						rt += f.Name
						typ = nil
					case *ast.ArrayType:
						rt += "[]"
						typ = f.Elt
					case *ast.StarExpr:
						rt += "*"
						typ = f.X
					default:
						log.Fatalf("unexpected field type: %s %T", field.Names[0], f)
					}
				}
				fields = append(fields, FieldInfo{
					FieldName: field.Names[0].String(),
					FieldType: rt,
				})
			}
		}
	}
	return fields
}
