/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

package org.jboss.internal.soa.esb.message.format.xml;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Map.Entry;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import org.jboss.internal.soa.esb.util.stax.ElementContent;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.internal.soa.esb.util.stax.TextElement;
import org.jboss.soa.esb.message.Attachment;

/**
 * Messages may contain attachments that do not appear in the main payload body.
 * For example, binary document formats, zip files etc.
 * 
 * @author Mark Little
 */

public class AttachmentImpl extends ElementContent implements Attachment
{
        /**
         * Default constructor.
         */
        public AttachmentImpl()
        {
        }
        
        /**
         * Construct the attachments from the input stream.
         * 
         * @param in The input stream.
         * @throws XMLStreamException For errors during parsing.
         */
        public AttachmentImpl(final XMLStreamReader in)
            throws XMLStreamException
        {
            parse(in) ;
        }
        
	public Object get(String name)
	{
		return unwrap(_table.get(name));
	}

	public Object put(String name, Object value)
	{
        assertValidAttachment(value);
		return unwrap(_table.put(name, new SerializedValueImpl((Serializable) value)));
    }

    public Object remove(String name)
	{
		return unwrap(_table.remove(name));
	}

	public String[] getNames()
	{
		return _table.keySet().toArray(new String[0]);
	}

	public Object itemAt(int index) throws IndexOutOfBoundsException
	{
		return unwrap(_list.get(index));
	}

	public Object removeItemAt(int index) throws IndexOutOfBoundsException
	{
		return unwrap(_list.remove(index));
	}

	public Object replaceItemAt(int index, Object value)
			throws IndexOutOfBoundsException
	{
        assertValidAttachment(value);
        return unwrap(_list.set(index, new SerializedValueImpl((Serializable) value)));
	}

	public void addItem(Object value)
	{
        assertValidAttachment(value);
        _list.add(new SerializedValueImpl((Serializable) value));
	}

	public void addItemAt(int index, Object value)
			throws IndexOutOfBoundsException
	{
        assertValidAttachment(value);
        _list.add(index, new SerializedValueImpl((Serializable) value));
	}

	public int getNamedCount()
	{
		return _table.size();
	}

	public int getUnnamedCount()
	{
		return _list.size();
	}

	public String toString()
	{
		return new StringBuilder().append("attachments: [ Named:").append(
				_table.toString()).append(", Unnamed:").append(_list.toString())
				.toString()+" ]";
	}

	/**
	 * Write the child content of the element.
         * @param out The output stream.
         * @throws XMLStreamException For errors during output.
         */
	@Override
	protected void writeChildContent(XMLStreamWriter out)
	        throws XMLStreamException
	{
            for (SerializedValueImpl value : _list)
            {
                final TextElement unnamed = new TextElement(value.getSerialisedForm()) ;
                StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_ATTACHMENT_UNNAMED, unnamed) ;
            }
            
            for (Entry<String, SerializedValueImpl> entry : _table.entrySet())
            {
                final TextElement named = new TextElement(entry.getValue().getSerialisedForm()) ;
                final String origNamedURI = StreamHelper.writeStartElement(out, XMLUtil.ESB_QNAME_ATTACHMENT_NAMED) ;
                StreamHelper.writeAttribute(out, XMLUtil.ESB_QNAME_ATTRIBUTE_ATTACHMENT_NAMED_NAME, entry.getKey()) ;
                named.writeContent(out) ;
                StreamHelper.writeEndElement(out, XMLUtil.ESB_QNAME_ATTACHMENT_NAMED.getPrefix(), origNamedURI) ;
            }
	}
	    
        /**
         * Add the element.
         * @param in The current input stream.
         * @param elementName The qualified element name.
         * @throws XMLStreamException For errors during parsing.
         */
        protected void putElement(final XMLStreamReader in,
            final QName elementName)
            throws XMLStreamException
        {
            if (XMLUtil.ESB_QNAME_ATTACHMENT_UNNAMED.equals(elementName))
            {
                final TextElement unnamed = new TextElement(in) ;
                final SerializedValueImpl value = new SerializedValueImpl(unnamed.getText());
                _list.add(value);
            }
            else if (XMLUtil.ESB_QNAME_ATTACHMENT_NAMED.equals(elementName))
            {
                final String name = StreamHelper.getAttributeValue(in, XMLUtil.ESB_QNAME_ATTRIBUTE_ATTACHMENT_NAMED_NAME) ;
                final TextElement valueElement = new TextElement(in) ;
                final SerializedValueImpl value = new SerializedValueImpl(valueElement.getText()) ;
                _table.put(name, value) ;
            }
            else
            {
                throw new XMLStreamException("Unexpected element name: " + elementName) ;
            }
        }
	
    private void assertValidAttachment(Object attachment) {
        if(attachment == null) {
            throw new IllegalArgumentException("Cannot set null object as a message attachment.  Attachment objects must be non-null and Serializable.");
        } else if(!(attachment instanceof Serializable)) {
            throw new IllegalArgumentException("Cannot set an Object of type '" + attachment.getClass().getName() + "' as a message attachment.  Attachment objects must be Serializable.");
        }
    }

    private Object unwrap(final SerializedValueImpl value)
    {
        return (value == null ? null : value.getValue()) ;
    }

	private ArrayList<SerializedValueImpl> _list = new ArrayList<SerializedValueImpl>();
	private Hashtable<String, SerializedValueImpl> _table = new Hashtable<String, SerializedValueImpl>();
}
