/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.internal.soa.esb.rosetta.pooling;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import javax.jms.Connection;
import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import javax.naming.Context;
import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;
import org.jboss.soa.esb.helpers.NamingContextException;
import org.jboss.soa.esb.helpers.NamingContextPool;

import com.arjuna.common.util.propertyservice.PropertyManager;

/**
 * Interface that needs to be implemented to provide pool of connections.
 * @see DefaultConnectionPoolImpl
 * Default implementation of Connection Pool
 * @author kstam
 * @author <a href="mailto:daniel.bevenius@gmail.com">Daniel Bevenius</a>				
 * Date: March 10, 2007
 */
public class JmsConnectionPool
{
	private static final int DEFAULT_POOL_SIZE = 20;
	private static final int DEFAULT_SLEEP = 30;
	
	private static int CONFIGURED_POOL_SIZE = DEFAULT_POOL_SIZE;
	private static int CONFIGURED_SLEEP = DEFAULT_SLEEP;
	
    /** Maximum number of Sessions that will be created in this pool */
    private int MAX_SESSIONS = DEFAULT_POOL_SIZE;    //TODO Make this manageable
    
    /** Time to sleep when trying to get a session. */
    private int SLEEP_TIME = DEFAULT_SLEEP;
    
    /** Number of free sessions in the pool that can be given out. Indexed by session key */
    private Map<Integer,ArrayList<Session>> freeSessionsMap = new HashMap<Integer,ArrayList<Session>>();
    
    /** Number of session that are currently in use. Indexed by session key mode */
    private Map<Integer,ArrayList<Session>> inUseSessionsMap = new HashMap<Integer,ArrayList<Session>>();
    
    /** Reference to a Queue or Topic Connection, we only need one per pool */
    protected Connection jmsConnection = null;
    
    /** The Indentifier of the pool */
    private Map<String, String> poolKey;
    
    /** Logger */
    private Logger logger = Logger.getLogger(this.getClass());
    
    /**
     * Contructor of the pool.
     * 
     */
    public JmsConnectionPool(Map<String, String> poolKey) 
    {
    	this(poolKey, JmsConnectionPool.CONFIGURED_POOL_SIZE, JmsConnectionPool.CONFIGURED_SLEEP);
    }
    
    public JmsConnectionPool(Map<String, String> poolKey, int poolSize, int sleepTime) 
    {
        this.poolKey = poolKey;
        
        MAX_SESSIONS = poolSize;
        SLEEP_TIME = sleepTime;
        
        freeSessionsMap.put(Session.AUTO_ACKNOWLEDGE, new ArrayList<Session>() );
        freeSessionsMap.put(Session.CLIENT_ACKNOWLEDGE, new ArrayList<Session>() );
        freeSessionsMap.put(Session.DUPS_OK_ACKNOWLEDGE, new ArrayList<Session>() );
        
        inUseSessionsMap.put(Session.AUTO_ACKNOWLEDGE, new ArrayList<Session>() );
        inUseSessionsMap.put(Session.CLIENT_ACKNOWLEDGE, new ArrayList<Session>() );
        inUseSessionsMap.put(Session.DUPS_OK_ACKNOWLEDGE, new ArrayList<Session>() );
    }
   
    /**
     * This is where we create the sessions. 
     * 
     * @param poolKey
     * @throws NamingException
     * @throws JMSException
     * @throws ConnectionException
     */
    private  synchronized void addAnotherSession(Map<String, String> poolKey, final int acknowledgeMode)
    throws NamingException, JMSException, ConnectionException, NamingContextException
    {
        String destinationType = poolKey.get(JMSEpr.DESTINATION_TYPE_TAG);

        //Setup a connection if we don't have one
        if (jmsConnection==null) {
            JmsConnectionPoolContainer.addToPool(poolKey, this);
            logger.debug("Creating a JMS Connection for poolKey : " + poolKey);
            Properties jndiEnvironment = JmsConnectionPoolContainer.getJndiEnvironment(poolKey);
            Context jndiContext = NamingContextPool.getNamingContext(jndiEnvironment);
            try {
                String connectionFactoryString = poolKey.get(JMSEpr.CONNECTION_FACTORY_TAG);
                Object factoryConnection=null;

                try
                {
                    factoryConnection = jndiContext.lookup(overrideName(connectionFactoryString));
                } catch (NamingException ne) {
                    logger.info("Received NamingException, refreshing context.");
                    jndiContext = NamingContextPool.replaceNamingContext(jndiContext, JmsConnectionPoolContainer.getJndiEnvironment(poolKey));
                    factoryConnection = jndiContext.lookup(connectionFactoryString);
                }
                final String username = poolKey.get( JMSEpr.JMS_SECURITY_PRINCIPAL_TAG );
                final String password = poolKey.get( JMSEpr.JMS_SECURITY_CREDENTIAL_TAG );
                boolean useJMSSecurity = (username != null && password != null);
                logger.debug( "JMS Security principal [" + username + "] using JMS Security : " + useJMSSecurity );
                
                if (JMSEpr.QUEUE_TYPE.equals(destinationType)) {
                    QueueConnectionFactory factory = (QueueConnectionFactory) factoryConnection;
                    jmsConnection = useJMSSecurity ? factory.createQueueConnection(username,password): factory.createQueueConnection();
                } else {
                    TopicConnectionFactory factory = (TopicConnectionFactory) factoryConnection;
                    jmsConnection = useJMSSecurity ? factory.createTopicConnection(username,password): factory.createTopicConnection();
                }
                
                addExceptionListener();
                
                jmsConnection.start();
            } finally {
                NamingContextPool.releaseNamingContext(jndiContext) ;
            }
        }
        final boolean transacted = Boolean.valueOf(poolKey.get(JMSEpr.TRANSACTED_TAG));
        
        //Create a new Session
        ArrayList<Session> freeSessions = freeSessionsMap.get( acknowledgeMode );

        if (JMSEpr.QUEUE_TYPE.equals(destinationType)) {
            QueueSession session = ((QueueConnection)jmsConnection).createQueueSession(transacted,acknowledgeMode);
                    
            freeSessions.add(session);
        } else if (JMSEpr.TOPIC_TYPE.equals(destinationType)) {
            TopicSession session = ((TopicConnection) jmsConnection).createTopicSession(transacted,acknowledgeMode);
            freeSessions.add(session);
        } else {
            throw new ConnectionException("Unknown destination type");
        }
        logger.debug("Number of Sessions in the pool with acknowledgeMode: " + acknowledgeMode + " is now " + getSessionsInPool(acknowledgeMode));
    }

    /**
     *  This method can be called whenever a connection is needed from the pool.
     *  
     * @return Connection to be used
     * @throws ConnectionException
     */
    public synchronized Session getSession(final int acknowledgeMode) throws NamingException, JMSException, ConnectionException
    {
        final long end = System.currentTimeMillis() + (SLEEP_TIME * 1000) ;
        boolean emitExpiry = logger.isDebugEnabled() ;
        for(;;) {
        	
        	ArrayList<Session> freeSessions = freeSessionsMap.get(acknowledgeMode );
        	ArrayList<Session> inUseSessions = inUseSessionsMap.get(acknowledgeMode);
            if (freeSessions.size() > 0)
            {
                if (logger.isDebugEnabled()) {
                    logger.debug("Returning session, poolsize=" + getSessionsInPool() 
                            + ", maxsize=" + MAX_SESSIONS 
                            + ", number of pools=" + JmsConnectionPoolContainer.getNumberOfPools());
                }
                final Session session = freeSessions.remove(freeSessions.size()-1);
                inUseSessions.add(session);
                return session ;
            } else if (inUseSessions.size()<MAX_SESSIONS) {
                try {
                    addAnotherSession(poolKey,acknowledgeMode);
                } catch (final NamingContextException nce) {
                    throw new ConnectionException("Unexpected exception accessing Naming Context", nce) ;
                }
                
                continue ;
            } else {
                if (emitExpiry)
                {
                    logger.debug("The connection pool was exhausted, waiting for a session to be released.") ;
                    emitExpiry = false ;
                }
                final long now = System.currentTimeMillis() ;
                final long delay = (end - now) ;
                if (delay <= 0)
                {
                    throw new ConnectionException("Could not obtain a JMS connection from the pool after "+SLEEP_TIME+"s.");
                }
                else
                {
                    try
                    {
                        wait(delay) ;
                    }
                    catch (final InterruptedException ie) {} // ignore
                }
            }
        }
    }
    /**
     *  This method can be called whenever a Queue Session is needed from the pool.
     * @return
     * @throws NamingException
     * @throws JMSException
     * @throws ConnectionException
     */
    public QueueSession getQueueSession() throws NamingException, JMSException, ConnectionException
    {
        return (QueueSession) getQueueSession(Session.AUTO_ACKNOWLEDGE);
    }
    
    public QueueSession getQueueSession(final int acknowledgeMode) throws NamingException, JMSException, ConnectionException
    {
        return (QueueSession) getSession(acknowledgeMode);
    }
    
    /**
     * This method can be called whenever a Topic Session is needed from the pool.
     * @return
     * @throws NamingException
     * @throws JMSException
     * @throws ConnectionException
     */
    public TopicSession getTopicSession() throws NamingException, JMSException, ConnectionException
    {
        return (TopicSession) getTopicSession(Session.AUTO_ACKNOWLEDGE);
    }
    
    public TopicSession getTopicSession(final int acknowledgeMode) throws NamingException, JMSException, ConnectionException
    {
        return (TopicSession) getSession(acknowledgeMode);
    }

    /**
     * This method closes an open connection and returns the connection to the pool.
     * @param sessionToClose The connection to be returned to the pool.
     * @throws SQLException
     */
    public  synchronized void closeSession(Session sessionToClose){
		try
		{
        	ArrayList<Session> sessions = freeSessionsMap.get(sessionToClose.getAcknowledgeMode());
        	if ( sessions != null )
	        	sessions.add(sessionToClose);
        	
            releaseSession(sessionToClose) ;
		} catch (JMSException e)
		{
			logger.error("JMSException while calling getAcknowledgeMode", e);
		}
    }
    
    /**
     * This method closes an open session without returning it to the pool.
     * @param sessionToClose The session to be returned to the pool.
     * @throws SQLException
     */
    public synchronized void releaseSession(final Session sessionToClose) {
    	try
		{
			ArrayList<Session> inUseSessions = inUseSessionsMap.get(sessionToClose.getAcknowledgeMode());
			if ( inUseSessions != null )
	            inUseSessions.remove(sessionToClose);
            notifyAll() ;
		} catch (JMSException e)
		{
			logger.error("JMSException while calling getAcknowledgeMode", e);
		}
    }

    /**
     * This method is called when the pool needs to destroyed. It closes all open sessions
     * and the connection and removes it from the container's poolMap.
     */
    public synchronized void removeSessionPool()
    {
        freeSessionsMap.get(Session.AUTO_ACKNOWLEDGE).clear() ;
        freeSessionsMap.get(Session.CLIENT_ACKNOWLEDGE).clear() ;
        freeSessionsMap.get(Session.DUPS_OK_ACKNOWLEDGE).clear() ;
        
        inUseSessionsMap.get(Session.AUTO_ACKNOWLEDGE).clear() ;
        inUseSessionsMap.get(Session.CLIENT_ACKNOWLEDGE).clear() ;
        inUseSessionsMap.get(Session.DUPS_OK_ACKNOWLEDGE).clear() ;
        
        logger.debug("Emptied the session pool now closing the connection to the factory.");
        if (jmsConnection!=null) {
            try {
                jmsConnection.close();
            } catch (final Exception ex) {} // ignore
            jmsConnection=null;
        }
        JmsConnectionPoolContainer.removePool(poolKey);
    }
    /**
     * Gets the total number of sessions in the pool regardless of the acknowlede mode
     * used when creating the sessions.
     * @return the session pool size
     */
    public int getSessionsInPool() {
    	int nrOfSessions = freeSessionsMap.get(Session.AUTO_ACKNOWLEDGE).size();
    	nrOfSessions += freeSessionsMap.get(Session.CLIENT_ACKNOWLEDGE).size();
    	nrOfSessions += freeSessionsMap.get(Session.DUPS_OK_ACKNOWLEDGE).size();
    	nrOfSessions += inUseSessionsMap.get(Session.AUTO_ACKNOWLEDGE).size();
    	nrOfSessions += inUseSessionsMap.get(Session.CLIENT_ACKNOWLEDGE).size();
    	nrOfSessions += inUseSessionsMap.get(Session.DUPS_OK_ACKNOWLEDGE).size();
    	return nrOfSessions;
    }
    
    /**
     * Returns the total nr of sessions for the specifed acknowledge mode
     * 
     * @param acknowledgeMode the acknowledge mode of sessions
     * @return
     */
    public int getSessionsInPool(final int acknowledgeMode) {
        return freeSessionsMap.get(acknowledgeMode).size() + inUseSessionsMap.get(acknowledgeMode).size();
    }
    
    /**
     * Get the number of free sessions created with the specified acknowledge mode
     * @param acknowledgeMode the acknowledge mode of sessions
     * @return int	the number of in use sessions
     */
    public int getFreeSessionsInPool(final int acknowledgeMode) {
        return freeSessionsMap.get(acknowledgeMode).size();
    }
    
    /**
     * Get the number of sessions that are in use and that were
     * created with the specified acknowledge mode
     * @param acknowledgeMode the acknowledge mode of sessions
     * @return int	the number of in use sessions
     */
    public int getInUseSessionsInPool(final int acknowledgeMode) {
        return inUseSessionsMap.get(acknowledgeMode).size();
    }
    
    protected String overrideName (String name) throws ConnectionException
    {
	return name;
    }
    
    protected void addExceptionListener () throws JMSException, ConnectionException
    {
        jmsConnection.setExceptionListener(new ExceptionListener() {
            public void onException(JMSException arg0)
            {
                removeSessionPool() ;
            }
        }) ;
    }
    
    static
    {
    	PropertyManager prop = ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE);
    	String value = prop.getProperty(Environment.JMS_CONNECTION_POOL_SIZE);
    	
    	if (value != null)
    	{
    		try
    		{
    			CONFIGURED_POOL_SIZE = Integer.parseInt(value);
    		}
    		catch (NumberFormatException ex)
    		{
    			ex.printStackTrace();
    		}
    	}
    	
    	value = prop.getProperty(Environment.JMS_SESSION_SLEEP);
    	
    	if (value != null)
    	{
    		try
    		{
    			CONFIGURED_SLEEP = Integer.parseInt(value);
    		}
    		catch (NumberFormatException ex)
    		{
    			ex.printStackTrace();
    		}
    	}
    }
}
