/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.rosetta.pooling.xa;

import java.util.Map;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Session;
import javax.naming.NamingException;

import org.jboss.internal.soa.esb.rosetta.pooling.ConnectionException;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPool;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;

/**
 * One instance per global transaction.
 * 
 * @author marklittle
 * 
 */

public class XaJmsConnectionPool extends JmsConnectionPool
{
    public XaJmsConnectionPool(Map<String, String> poolKey)
    {
	super(poolKey);
    }

    public XaJmsConnectionPool(Map<String, String> poolKey, int poolSize,
	    int sleepTime)
    {
	super(poolKey, poolSize, sleepTime);
    }

    public boolean active ()
    {
	return _active;
    }

    public void terminated ()
    {
	_active = false;
    }

    /*
     * Make sure we return the same connection during the transaction.
     * @see org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPool#getSession(int)
     */
    public synchronized Session getSession (final int acknowledgeMode)
	    throws NamingException, JMSException, ConnectionException
    {
	if (_theSession == null)
	    _theSession = super.getSession(acknowledgeMode);

	return _theSession;
    }

    public synchronized void closeSession (Session sessionToClose)
    {
	// not within the scope of a transaction!
    }

    public synchronized void releaseSession (final Session sessionToClose)
    {
	// not within the scope of a transaction!
    }

    public synchronized void removeSessionPool()
    {
	// ignore and let Synchronization cleanup for us.
    }
    
    public synchronized void removeTxSessionPool()
    {
	super.closeSession(_theSession);
	_theSession = null;
	
	super.removeSessionPool();
    }
    
    protected String overrideName (String name) throws ConnectionException
    {
	try
	{
	    /*
                 * If there is a global transaction in place then make sure we
                 * grab the right connection, irrespective of the provider info.
                 */

	    if (TransactionStrategy.getTransactionStrategy(true)
		    .getTransaction() != null)
	    {
		if (("XAConnectionFactory".equals(name))
			|| ("ConnectionFactory".equals(name)))
		    return "java:/JmsXA";
	    }

	    return name;
	}
	catch (TransactionStrategyException ex)
	{
	    throw new ConnectionException(ex);
	}
    }

    protected void addExceptionListener () throws JMSException,
	    ConnectionException
    {
	/*
         * It is illegal to set ExceptionListeners if we are within a
         * transaction.
         */

	try
	{
	    TransactionStrategy txStrategy = TransactionStrategy
		    .getTransactionStrategy(true);

	    if ((txStrategy != null) && (txStrategy.getTransaction() == null))
	    {
		jmsConnection.setExceptionListener(new ExceptionListener()
		{
		    public void onException (JMSException arg0)
		    {
			removeSessionPool();
		    }
		});
	    }
	}
	catch (TransactionStrategyException ex)
	{
	    throw new ConnectionException(ex);
	}
    }

    private boolean _active = true;

    private Session _theSession = null;
}
