/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/*
 * Copyright (c) 2002, 2003, Arjuna Technologies Limited.
 *
 * PortReference.java
 */

package org.jboss.soa.esb.addressing;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/**
 * An implementation of a WS-Addressing EPR. It needs completely rewriting after
 * the interoperability workshop as it is not extensible. It's morphed with the changing WS-C/WS-T and
 * WS-Addr specifications and their quirks; now that WS-Addr is finalized the
 * old quirks no longer need to be supported so it's best to rewrite this from
 * scratch.
 * 
 * An instance of a PortReference represents a single element in WS-A.
 */

public class PortReference implements Cloneable
{
	public PortReference()
	{
	}

	public PortReference(String address)
	{
		setAddress(address);
	}

	public void setAddress(String address)
	{
		_address = address;
	}

	public String getAddress()
	{
		return _address;
	}

	public void addExtension(PortReference.Extension extension)
	{
		_extensions.add(extension);
	}

	// all extensions are added as the Arjuna namespace.

	public void addExtension(String value)
	{
		addExtension(XMLUtil.INSTANCE_IDENTIFIER_TAG, XMLUtil.JBOSSESB_PREFIX,
				XMLUtil.JBOSSESB_NAMESPACE_URI, value,
				Extension.REFERENCE_PROPERTIES);
	}
	
	public void removeExtension (String tag, String value)
	{
	    /*
	     * Need tag and value to make sure it's unique.
	     */
	    
	    _extensions.remove(new Extension(tag, XMLUtil.JBOSSESB_PREFIX, XMLUtil.JBOSSESB_NAMESPACE_URI, value));
	}
	
	public void addExtension(String tag, String value)
	{
		addExtension(tag, XMLUtil.JBOSSESB_PREFIX,
				XMLUtil.JBOSSESB_NAMESPACE_URI, value,
				Extension.REFERENCE_PROPERTIES);
	}

	/**
	 * Define the tag, prefix and namespace URI for the extension value. The
	 * parent is a refProperty.
	 */

	public void addExtension(String tag, String prefix, String uri, String value)
	{
		_extensions.add(new Extension(tag, prefix, uri, value,
				Extension.REFERENCE_PROPERTIES));
	}

	/**
	 * Define the tag, prefix and namespace URI for the extension value. The
	 * parent field defines whether the attribute has a refParam, refProp or
	 * neither as its parent.
	 */

	public void addExtension(String tag, String prefix, String uri,
			String value, int parent)
	{
		_extensions.add(new Extension(tag, prefix, uri, value, parent));
	}

        public void addExtensions(final List<Extension> extensions)
        {
            _extensions.addAll(extensions);
        }

	// placeholders only

	public void addPortType(String qName)
	{
	}

	public void addServiceName(String portName, String qName)
	{
	}

	public void addPolicy()
	{
	}

	public String getExtensionValue(String tag)
	{
		String extensionValue = null;
		Iterator iterator = _extensions.iterator();

		while (iterator.hasNext() && (extensionValue == null))
		{
			Extension extension = (Extension) iterator.next();

			if (tag.equals(extension.getTag()))
				extensionValue = extension.getValue();
		}

		return extensionValue;
	}

	public Iterator<Extension> getExtensions()
	{
		return _extensions.iterator();
	}

        public List<Extension> getAllExtensions()
        {
                return _extensions;
        }

	public void clearExtensions()
	{
		_extensions.clear();
	}

	/**
	 * Return a copy of this instance.
	 * 
	 * @return a field-wise copy.
	 */
	
	public PortReference copy ()
	{
		try
		{
			/*
			 * In general, try ry to avoid clone (http://www.artima.com/intv/blochP.html)
			 * But should be ok here because the invariants are known and field-wise
			 * copying should therefore be safe.
			 */
			final PortReference clone = (PortReference)clone() ;
			clone._extensions = new LinkedList<Extension>(_extensions) ;
			return clone;
		}
		catch (Exception ex)
		{
			return null;
		}
	}
	
	public String toString()
	{
		return "PortReference < " + _address + " >";
	}

	public String extendedToString()
	{
		String addr = "<" + XMLUtil.WSA_PREFIX + ":Address " + _address + "/>";

		Iterator extensions = getExtensions();

		while (extensions.hasNext())
		{
			Extension ext = (Extension) extensions.next();

			addr += ", <" + XMLUtil.WSA_PREFIX + ":"
					+ XMLUtil.REFERENCE_PROPERTIES_TAG + " " + ext.getPrefix()
					+ ":" + ext.getTag() + " : " + ext.getValue() + "/>";
		}

		return "PortReference < " + addr + " >";
	}

	/**
	 * Is this object equal to the specified parameter?
	 * 
	 * @param rhs
	 *            The rhs object.
	 * @return true if the specified object is equal, false otherwise.
	 */
	public boolean equals(final Object rhs)
	{
		if ((rhs != null) && (rhs.getClass() == getClass()))
		{
			// This should really only include reference property extensions
			final PortReference rhsPortReference = (PortReference) rhs;
			return (equalsObject(_address, rhsPortReference._address)
					&& equalsObject(_extensions, rhsPortReference._extensions)
					&& equalsObject(_portType, rhsPortReference._portType)
					&& equalsObject(_serviceName, rhsPortReference._serviceName) && equalsObject(
					_policies, rhsPortReference._policies));
		}
		return false;
	}

	/**
	 * Return the hash code for this object.
	 * 
	 * @return the hash code value.
	 */
	public int hashCode()
	{
		// Not checked for spread.
		return (objectHashCode(_address, 0x1)
				^ objectHashCode(_extensions, 0x2)
				^ objectHashCode(_portType, 0x4)
				^ objectHashCode(_serviceName, 0x8) ^ objectHashCode(_policies,
				0x10));
	}

	public static class Extension
	{
		public static final int REFERENCE_PROPERTIES = 0;
		public static final int REFERENCE_PARAMETERS = 1;
		public static final int NEITHER = 2;

		public Extension(String tag, String prefix, String uri)
		{
			this(tag, prefix, uri, null, REFERENCE_PROPERTIES);
		}

		public Extension(String tag, String prefix, String uri, String value)
		{
			this(tag, prefix, uri, value, REFERENCE_PROPERTIES);
		}

		public Extension(String tag, String prefix, String uri, String value,
				int parent)
		{
			_tag = tag;
			_prefix = prefix;
			_uri = uri;
			_value = value;
			_parent = parent;
			
			/*
			 * If XML was loaded via DOM1 then prefix and namespace won't work!
			 */
			
			if (_uri == null)
				_uri = XMLUtil.JBOSSESB_NAMESPACE_URI;
			
			if (_prefix == null)
				_prefix = XMLUtil.JBOSSESB_PREFIX;
			
			if (_tag.contains(":"))
			{
				int colon = _tag.indexOf(':');
				
				_tag = _tag.substring(colon+1);
			}
		}

		public int getParent()
		{
			return _parent;
		}

		public String getTag()
		{
			return _tag;
		}

		public String getPrefix()
		{
			return _prefix;
		}

		public String getURI()
		{
			return _uri;
		}

		public String getValue()
		{
			return _value;
		}

		public LinkedList getChildren()
		{
			return _extensions;
		}

		public HashMap getAttributes()
		{
			return _attributes;
		}

		public void addAttributes(HashMap props)
		{
			_attributes = props;
		}

		public void addChild(Extension child)
		{
			if (_extensions == null)
				_extensions = new LinkedList<Extension>();

			_extensions.add(child);
		}

		public String toString()
		{
			return new String("< " + _tag + ", " + _prefix + ", " + _uri + ", "
					+ _value + " >");
		}

		/**
		 * Is this object equal to the specified parameter (ignoring prefix)?
		 * 
		 * @param rhs
		 *            The rhs object.
		 * @return true if the specified object is equal, false otherwise.
		 */
		public boolean equals(final Object rhs)
		{
			if ((rhs != null) && (rhs.getClass() == getClass()))
			{
				final Extension rhsExtension = (Extension) rhs;
				return (equalsObject(_tag, rhsExtension._tag)
						&& equalsObject(_uri, rhsExtension._uri)
						&& equalsObject(_value, rhsExtension._value)
						&& equalsObject(_extensions, rhsExtension._extensions) && (_parent == rhsExtension._parent));
			}
			return false;
		}

		/**
		 * Return the hash code for this object.
		 * 
		 * @return the hash code value.
		 */
		public int hashCode()
		{
			// Not checked for spread.
			return (_parent ^ objectHashCode(_tag, 0x4)
					^ objectHashCode(_uri, 0x8) ^ objectHashCode(_value, 0x10) ^ objectHashCode(
					_extensions, 0x20));
		}

		private String _tag = null;

		private String _prefix = null;

		private String _uri = null;

		private String _value = null;

		private int _parent = NEITHER;

		private LinkedList<Extension> _extensions = null;

		private HashMap _attributes = null;

	}

	/**
	 * Get the hash code from the object or use the default if null.
	 * 
	 * @param obj
	 *            The object.
	 * @param defaultHashCode
	 *            The default hash code.
	 * @return The hash code.
	 */
	static int objectHashCode(final Object obj, final int defaultHashCode)
	{
		return (obj == null ? defaultHashCode : obj.hashCode());
	}

	/**
	 * Are the two objects equal?
	 * 
	 * @param lhs
	 *            The lhs object.
	 * @param rhs
	 *            The rhs object.
	 * @return true if equal or both null, false otherwise.
	 */
	static boolean equalsObject(final Object lhs, final Object rhs)
	{
		if (lhs == null)
		{
			return (rhs == null);
		} else
		{
			return lhs.equals(rhs);
		}
	}

	private String _address = null;
	private LinkedList<Extension> _extensions = new LinkedList<Extension>();
	private Extension _portType = null;
	private Extension _serviceName = null;
	private List _policies = null;

}
