/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.couriers;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.io.StreamCorruptedException;
import java.net.URISyntaxException;
import java.util.Properties;

import javax.xml.parsers.ParserConfigurationException;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.DeliverOnlyCourier;
import org.jboss.internal.soa.esb.couriers.PickUpOnlyCourier;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.addressing.util.DefaultReplyTo;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.util.ContextObjectInputStream;
import org.jboss.soa.esb.util.Util;
import org.xml.sax.SAXException;

public class CourierUtil
{
	private CourierUtil() {}

	/**
	 * Obtain a default replyToEPR (based on the toEpr) to which we can reply to.
	 *
	 * @deprecated Use DefaultReplyTo helper class instead.
	 *	
	 * @param toEpr - the toEPR which will be used a basis to generate the replyToEPR.
	 * @return EPR - the EPR to which the pickup will be delivered.
	 * @throws URISyntaxException,
	 *             CourierException
	 */
	public static EPR getDefaultReplyToEpr(EPR toEpr)
			throws CourierException, MalformedEPRException
	{
	    return DefaultReplyTo.getReplyTo(toEpr);
	}

	/**
	 * Obtain properties and values used in a JMS message selector
	 * @param selector  A valid JMS message selector
	 * @return Properties  A Properties object containing the keys and values in the message selector
	 * @throws ConfigurationException
	 * @ceprecated Use Util.propertiesFromSelector
	 */
	public static Properties propertiesFromSelector(String selector)
		throws ConfigurationException
	{
		Properties oRet = new Properties();
		if (!Util.isNullString(selector))
		{
			for (String sCurr : selector.split(","))
			{
				String[] sa = sCurr.split("=");
				if (sa.length != 2 || sa[1].charAt(0) != '\''
						|| sa[1].charAt(-1 + sa[1].length()) != '\'')
					throw new ConfigurationException("Illegal message selector syntax <"
							+ selector + ">  list of comma separated key='value' expected");
				KeyValuePair oNew = new KeyValuePair(sa[0], sa[1].substring(0,
						-1 + sa[1].length()).substring(1));
				oRet.put(oNew.getKey(), oNew.getValue());
			}
		}
		return oRet;
	}
	

	public static File messageToLocalFile(File directory, Message message)
			throws IOException, ParserConfigurationException
	{
		File tmpFile = File.createTempFile("EsbFileCourier_", ".__esbPart",
				directory);
		Serializable serial = Util.serialize(message);
		FileOutputStream writer = null;
		try
		{
			writer = new FileOutputStream(tmpFile);
			new ObjectOutputStream(writer).writeObject(serial);
			
			writer.flush();
		}
		finally
		{
			if (null != writer)
				writer.close();
		}
		return tmpFile;
	}

	public static Message messageFromLocalFile(File from)
			throws FileNotFoundException, IOException, ClassNotFoundException,
			ClassCastException, ParserConfigurationException, SAXException,
			CourierException
	{
		FileInputStream reader = null;
		Serializable serial = null;
		int retry = 10; // TODO magic number
		
		/*
		 * Just because a file is in the file system doesn't mean it
		 * has been completely written!
		 */
		
		while (retry > 0)
		{
			try
			{
				reader = new FileInputStream(from);
				serial = (Serializable) new ContextObjectInputStream(reader).readObject();
				return Util.deserialize(serial);
			}
			catch (StreamCorruptedException ex)
			{
				// file present but not ready to read - wait
				
				retry--;
			}
			catch (Exception e)
			{
				e.printStackTrace();
				
				_logger.debug(from+" "+e.toString());
				throw new CourierException(e);
			}
			finally
			{
				if (null != reader)
					reader.close();
			}
			
			try
			{
				Thread.sleep(1000);  // TODO magic number
			}
			catch (Exception ex)
			{
			}
		}
		
		throw new IOException();
	}

	public static byte[] bytesFromLocalFile(File from) throws IOException
	{
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		FileInputStream in = new FileInputStream(from);
		byte[] buff = new byte[1000];
		int iQ = 0;
		try
		{
			while ((iQ = in.read(buff)) >= 0)
				if (iQ > 0)
					out.write(buff, 0, iQ);
		}
		finally
		{
			if (null != in)
				in.close();
			if (null != out)
				out.close();
		}
		return out.toByteArray();
	}

	public static void bytesToLocalFile(byte[] bytes, File to)
			throws IOException
	{
		FileOutputStream out = null;
		try
		{
			out = new FileOutputStream(to);
			out.write(bytes);
		}
		finally
		{
			if (null != out)
				out.close();
		}
	}

	public static void deliverMessage(Message message)
			throws URISyntaxException, CourierException, MalformedEPRException
	{
		EPR toEpr = message.getHeader().getCall().getTo();
		Courier courier = CourierFactory.getCourier(toEpr);
                try
                {
                    courier.deliver(message);
                }
                finally
                {
                    CourierUtil.cleanCourier(courier) ;
                }
	}

    public static void cleanCourier (PickUpOnlyCourier courier)
    {
        if (null!=courier)
        {
            try
            {
                courier.cleanup() ;
            }
            catch (Exception e)
            {
                _logger.error("Problems invoking clean() Method for class "
                                +courier.getClass().getSimpleName(),e);
            }
        }
    }
    
    public static void cleanCourier (DeliverOnlyCourier courier)
    {
        if (null!=courier)
        {
            try
            {
                courier.cleanup() ;
            }
            catch (Exception e)
            {
                _logger.error("Problems invoking clean() Method for class "
                                +courier.getClass().getSimpleName(),e);
            }
        }
    }
    
    public static void cleanCourier (TwoWayCourier courier)
    {
        if (null!=courier)
        {
            try
            {
                courier.cleanup() ;
            }
            catch (Exception e)
            {
                _logger.error("Problems invoking clean() Method for class "
                                +courier.getClass().getSimpleName(),e);
            }
        }
    }
	    

	protected static Logger _logger = Logger.getLogger(CourierUtil.class);
}
