/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.http;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.http.configurators.HttpProtocol;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnectionManager;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.params.HttpConnectionManagerParams;

import java.util.Properties;
import java.io.IOException;

/**
 * {@link org.apache.commons.httpclient.HttpClient} factory class.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class HttpClientFactory {

    /**
     * Target host url.
     */
    public static final String TARGET_HOST_URL = "target-host-url";
    /**
     * Max total connections.
     */
    public static final String MAX_TOTAL_CONNECTIONS = "max-total-connections";

    /**
     * Factory method for creating the {@link org.apache.commons.httpclient.HttpClient}
     * instance.
     *
     * @param properties {@link org.apache.commons.httpclient.HttpClient} creation properties.
     * @return The {@link org.apache.commons.httpclient.HttpClient} instance.
     * @throws ConfigurationException Invalid configuration properties set.
     */
    public static HttpClient createHttpClient(Properties properties) throws ConfigurationException {
        AssertArgument.isNotNull(properties, "properties");

        final HttpConnectionManager connectionManager = new MultiThreadedHttpConnectionManager() ;
        final String maxTotalConnections = properties.getProperty(MAX_TOTAL_CONNECTIONS) ;
        if (maxTotalConnections != null) {
            final HttpConnectionManagerParams params = new HttpConnectionManagerParams() ;
            try {
                params.setMaxTotalConnections(Integer.parseInt(maxTotalConnections)) ;
            } catch (final NumberFormatException nfe) {
                throw new ConfigurationException("Invalid " + MAX_TOTAL_CONNECTIONS + " property: " + maxTotalConnections) ;
            }
        }
        
        HttpClient httpClient = new HttpClient(connectionManager);
        String fileConfig = (String) properties.remove("file");
        String[] configurators;

        if(fileConfig != null) {
            try {
                properties.load(StreamUtils.getResource(fileConfig));
            } catch (IOException e) {
                throw new ConfigurationException("Failed to load HttpClient config '" + fileConfig + "'.");
            }
        }

        configurators = properties.getProperty("configurators", HttpProtocol.class.getName()).split(",");
        for(String configuratorClass : configurators) {
            Configurator configurator = createConfigurator(configuratorClass.trim());
            configurator.configure(httpClient, properties);
        }

        return httpClient;
    }

    private static Configurator createConfigurator(String configuratorClass) throws ConfigurationException {
        try {
             return (Configurator) ClassUtil.forName(configuratorClass, HttpClientFactory.class).newInstance();
        } catch (ClassCastException e) {
            throw new ConfigurationException("Class [" + configuratorClass + "] must extend [" + Configurator.class.getName() + "].", e);
        } catch (ClassNotFoundException e) {
            if(!configuratorClass.startsWith(HttpProtocol.class.getPackage().getName())) {
                return createConfigurator(HttpProtocol.class.getPackage().getName() + "." + configuratorClass);
            }
            throw new ConfigurationException("Configurator implementation class [" + configuratorClass + "] not found in classpath.", e);
        } catch (InstantiationException e) {
            throw new ConfigurationException("Failed to instantiate Configurator implementation class [" + configuratorClass + "].", e);
        } catch (IllegalAccessException e) {
            throw new ConfigurationException("Failed to instantiate Configurator implementation class [" + configuratorClass + "].", e);
        }
    }

    /**
     * Shutdown http client and associated connections.
     * @param httpclient The http client
     */
    public static void shutdown(final HttpClient httpclient) {
        if (httpclient != null) {
            final HttpConnectionManager connectionManager = httpclient.getHttpConnectionManager();
            if (connectionManager instanceof MultiThreadedHttpConnectionManager) {
                final MultiThreadedHttpConnectionManager multiThreadedHttpConnectionManager = (MultiThreadedHttpConnectionManager)connectionManager;
                multiThreadedHttpConnectionManager.shutdown();
            }
        }
    }
}
