/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners;

import org.jboss.soa.esb.listeners.lifecycle.AbstractManagedLifecycle;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.listeners.message.ActionProcessingPipeline;
import org.jboss.soa.esb.schedule.ScheduledEventListener;
import org.jboss.soa.esb.schedule.SchedulingException;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.Initializable;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.message.Message;

/**
 * ESB Schedule listener.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ScheduleListener extends AbstractManagedLifecycle implements ScheduledEventListener {

    /**
     * 
     */
    private static final long serialVersionUID = 1L;
    /**
     * The action pipeline.
     */
    private ActionProcessingPipeline pipeline;
    /**
     * Event message processor. 
     */
    private Initializable eventProcessor;
    /**
     * The transaction strategy.
     */
    private final TransactionStrategy transactionStrategy ;

    /**
     * Construct the managed lifecycle.
     *
     * @param config The configuration associated with this instance.
     * @throws org.jboss.soa.esb.ConfigurationException
     *          for configuration errors during initialisation.
     */
    public ScheduleListener(ConfigTree config) throws ConfigurationException {
        super(config);

        // Construct the message composer instance...
        String eventProcessorClass = config.getRequiredAttribute("event-processor");
        try {
            eventProcessor = (Initializable) ClassUtil.forName(eventProcessorClass, ScheduleListener.class).newInstance();
        } catch (ClassCastException e) {
            throwBadImplException(eventProcessorClass);
        } catch (ClassNotFoundException e) {
            throw new ConfigurationException("Event Processor class [" + eventProcessorClass + "] not found in classpath.", e);
        } catch (InstantiationException e) {
            throw new ConfigurationException("Failed to instantiate Event Processor class [" + eventProcessorClass + "].", e);
        } catch (IllegalAccessException e) {
            throw new ConfigurationException("Failed to instantiate Event Processor class [" + eventProcessorClass + "].", e);
        }

        final boolean transacted = config.getBooleanAttribute(ListenerTagNames.TRANSACTED_TAG, false) ;
        transactionStrategy = TransactionStrategy.getTransactionStrategy(transacted) ;
        
        if(!(eventProcessor instanceof ScheduledEventListener) && !(eventProcessor instanceof ScheduledEventMessageComposer)) {
            throwBadImplException(eventProcessorClass);
        }
        eventProcessor.initialize(config);

        // Only construct a pipeline if it's a composer...
        if(eventProcessor instanceof ScheduledEventMessageComposer) {
            // Construct the Message processing pipeline instance...
            pipeline = new ActionProcessingPipeline(getConfig());
            pipeline.initialise();
        }        
    }

    private void throwBadImplException(String eventProcessorClass) throws ConfigurationException {
        throw new ConfigurationException("The 'even-processor' class '" + eventProcessorClass +
                "' must implement one of '" + ScheduledEventMessageComposer.class.getName() + "' or '" +
                ScheduledEventListener.class.getName() + "'.");
    }

    /**
     * On schedule event processor implementation.
     * @throws SchedulingException Error processing scheduled event.
     */
    public void onSchedule() throws SchedulingException {
        Message message;

        try {
            transactionStrategy.begin() ;
            boolean rollbackOnly = true ;
            try {
                if(eventProcessor instanceof ScheduledEventMessageComposer) {
                    ScheduledEventMessageComposer composer = (ScheduledEventMessageComposer)eventProcessor;
                    message = composer.composeMessage();
                    if(message != null)
                    {
        	            pipeline.process(message);
        	            composer.onProcessingComplete(message);
                    }
                } else {
                    ((ScheduledEventListener)eventProcessor).onSchedule();
                }
                rollbackOnly = false ;
            } finally {
                if (rollbackOnly) {
                    transactionStrategy.rollbackOnly() ;
                }
                transactionStrategy.terminate() ;
            }
        } catch (final TransactionStrategyException tse) {
            throw new SchedulingException("Unexpected transaction strategy exception", tse) ;
        }
    }

    protected void doInitialise() throws ManagedLifecycleException {
    }

    public void initialize(ConfigTree config) throws ConfigurationException {
        // TODO Convert lifecycle code to use the Initializable interface.
    }

    public void uninitialize() {
    }

    protected void doStart() throws ManagedLifecycleException {
    }

    protected void doStop() throws ManagedLifecycleException {
    }

    protected void doDestroy() throws ManagedLifecycleException {
        eventProcessor.uninitialize();
        if(pipeline != null) {
            pipeline.destroy() ;
        }
    }
}
