/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config;

import java.util.List;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.config.xbeanmodel.FsListenerDocument.FsListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel.FtpListenerDocument.FtpListener;
import org.jboss.soa.esb.listeners.config.Generator.XMLBeansModel;
import org.jboss.soa.esb.listeners.config.xbeanmodel.JmsListenerDocument.JmsListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ServiceDocument.Service;
import org.jboss.soa.esb.listeners.config.xbeanmodel.SqlListenerDocument.SqlListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ScheduledListenerDocument.ScheduledListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel.Listener;
import org.jboss.soa.esb.listeners.config.xbeanmodel.JbrListenerDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel.GroovyListenerDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel.AbstractScheduledListener;
import org.jboss.soa.esb.listeners.config.mappers.*;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * ESB Aware Listeners Configuration Generator.
 * <p/>
 * Generates a "ConfigTree" style configuration for the ESB Aware Listeners. 
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ESBAwareGenerator {

    private static final Logger logger = Logger.getLogger(ESBAwareGenerator.class);

    /**
	 * XMLBeans based configuration model instance.
	 */
	@SuppressWarnings("unused")
	private XMLBeansModel model;
	
	/**
	 * Public constructor.
	 * @param model XMLBeans based configuration model instance.
	 */
	public ESBAwareGenerator(XMLBeansModel model) {
		this.model = model;
	}

	/**
	 * Generate the configuration DOM for the target Server.
	 * @return The configuration DOM.
	 * @throws ConfigurationException Error creating configuration. 
	 */
	public Document generate() throws ConfigurationException {
		Document doc = YADOMUtil.createDocument();
		Element root;
		
		doc.appendChild(doc.createComment("\n\tNOTE: DO NOT MODIFY\n\tThis file was auto-generated.\n"));
		root = YADOMUtil.addElement(doc, "jbossesb-listeners");
		root.setAttribute("parameterReloadSecs", model.getParameterReloadSecs());
		
		// Iterate over all the ESB Aware listner configs and map them to the listener configuration...
		List<Listener> listeners = model.getESBAwareListeners();
		for(Listener listener : listeners) {
			addESBAwareConfig(root, listener);
		}
		
		return doc;
	}

	/**
	 * Add a single ESB Aware Listener configuration node to configuration root. 
	 * @param root Configuration root node.
	 * @param listener The ESB Aware Listener configuration to be added.
	 * @throws ConfigurationException Invalid listener configuration.
	 */
	private void addESBAwareConfig(Element root, Listener listener) throws ConfigurationException {
		Element listenerConfigTree;
		Service listenerService;
		
		// Of course we could do the following reflectively if we really want to - TODO perhaps!!
		if(listener instanceof JmsListener) {
			listenerConfigTree = JmsListenerMapper.map(root, (JmsListener)listener, model);
        } else if (listener instanceof ScheduledListener) {
            listenerConfigTree = ScheduledListenerMapper.map(root, (ScheduledListener) listener);
        } else if(listener instanceof FsListener) {
         listenerConfigTree = FsListenerMapper.map(root, (FsListener)listener, model);
		} else if (listener instanceof FtpListener) {
			listenerConfigTree = FtpListenerMapper.map(root, (FtpListener)listener, model);
		} else if (listener instanceof SqlListener) {
			listenerConfigTree = SqlListenerMapper.map(root, (SqlListener)listener, model);
        } else if (listener instanceof JbrListenerDocument.JbrListener) {
            logger.error("JbossRemoting listener only supported in Gateway mode.");
            return;
        } else if (listener instanceof GroovyListenerDocument.GroovyListener) {
            listenerConfigTree = GroovyListenerMapper.map(root, (GroovyListenerDocument.GroovyListener)listener, model);
		} else {
			listenerConfigTree = UntypedListenerMapper.map(root, listener, model);
		}

        if (listener instanceof AbstractScheduledListener) {
            ScheduleMapper.map(listenerConfigTree, (AbstractScheduledListener) listener);
        }

        // Locate the Service to which the listener is bound...
		listenerService = model.getService(listener);
		
		// Map the actions and notiications...
		ActionMapper.map(listenerConfigTree, listenerService, model);
		NotificationMapper.map(listenerConfigTree, listenerService, model);
	}
}
