/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners.gateway;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.MessageConsumer;
import javax.jms.Queue;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicSession;
import javax.naming.Context;
import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.rosetta.pooling.ConnectionException;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPool;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPoolContainer;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.couriers.Courier;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierFactory;
import org.jboss.soa.esb.couriers.CourierUtil;
import org.jboss.soa.esb.filter.FilterManager;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.NamingContextException;
import org.jboss.soa.esb.helpers.NamingContextPool;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.ListenerUtil;
import org.jboss.soa.esb.listeners.RegistryUtil;
import org.jboss.soa.esb.listeners.lifecycle.AbstractThreadedManagedLifecycle;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleThreadState;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;
import org.jboss.soa.esb.util.ClassUtil;

public class JmsGatewayListener extends AbstractThreadedManagedLifecycle {
    /**
     * serial version uid for this class
     */
    private static final long serialVersionUID = 5070422864110923930L;

    public JmsGatewayListener(ConfigTree listenerConfig)
            throws ConfigurationException {
        super(listenerConfig);
        _config = listenerConfig;
        checkMyParms();
    } // __________________________________

    /**
     * Handle the initialisation of the managed instance.
     *
     * @throws ManagedLifecycleException for errors while initialisation.
     */
    protected void doInitialise() throws ManagedLifecycleException {
        try {
            _targetEprs = RegistryUtil.getEprs(_targetServiceCategory,
                    _targetServiceName);
            if (null == _targetEprs || _targetEprs.size() < 1)
                throw new ManagedLifecycleException("EPR <"
                        + _targetServiceName + "> not found in registry");
       } catch (ServiceNotFoundException snfe) {
        throw new ManagedLifecycleException("EPR <" + _targetServiceName + " "
                + _targetServiceName + "> not found in registry");
       }
        catch (final RegistryException re) {
            throw new ManagedLifecycleException(
                    "Unexpected registry exception", re);
        }

        try {
            prepareMessageReceiver();
        }
        catch (final ConnectionException ce) {
            throw new ManagedLifecycleException(
                    "Unexpected connection exception from prepareMessageReceiver",
                    ce);
        }
        catch (final JMSException jmse) {
            throw new ManagedLifecycleException(
                    "Unexpected JMS error from prepareMessageReceiver", jmse);
        }
        catch (final ConfigurationException ce) {
            throw new ManagedLifecycleException(
                    "Unexpected configuration exception from prepareMessageReceiver",
                    ce);
        } catch (final NamingContextException nce) {
            throw new ManagedLifecycleException(
                    "Unexpected naming context exception from prepareMessageReceiver",
                    nce);
        }

        if (_serviceName != null) {
            try {
                RegistryUtil.register(_config, _myEpr);
            }
            catch (final RegistryException re) {
                throw new ManagedLifecycleException(
                        "Unexpected error during registration for epr "
                                + _myEpr, re);
            }
        }
    }

    /**
     * Execute on the thread.
     */
    protected void doRun() {
        if (_logger.isDebugEnabled()) {
            _logger.debug("run() method of " + this.getClass().getSimpleName()
                    + " started on thread " + Thread.currentThread().getName());
        }

        while (isRunning()) {
            javax.jms.Message msgIn = receiveOne();
            
            if (null != msgIn) {
                try {
                    Object obj = _processMethod.invoke(_composer, new Object[] {msgIn});
                    // commit and acknowledge the reception of the message
                    // this is done after extracting the content of the JMS Message.
                    if ( jmsSession != null && jmsSession.getTransacted() )
                    	jmsSession.commit();
                    	
                    if (null == obj) {
                        _logger.warn("Action class method <"
                                + _processMethod.getName()
                                + "> returned a null object");
                        continue;
                    }
                    // try to deliverAsync the composed message, using the
                    // appropriate courier
                    // to the target service

                    Map<String, Object> params = new HashMap<String, Object>();

                    params.put(Environment.GATEWAY_CONFIG, _config);

                    obj = FilterManager.getInstance().doOutputWork((Message) obj, params);

                    try {
                        boolean bSent = false;
                        for (EPR current : _targetEprs) {
                            _courier = CourierFactory.getCourier(current);
                            try {
                                if (_courier.deliver((Message) obj)) {
                                    bSent = true;
                                    break;
                                }
                            }
                            finally {
                                CourierUtil.cleanCourier(_courier);
                            }
                        }
                        if (!bSent) {
                            String text = "Target service <"
                                    + _targetServiceCategory + ","
                                    + _targetServiceName
                                    + "> is not registered";
                            throw new Exception(text);
                        }
                    }
                    catch (ClassCastException e) {
                        _logger.error("Action class method <"
                                + _processMethod.getName()
                                + "> returned a non Message object", e);
                        
	                    rollbackJMSTransaction();
                        continue;
                    }
                    catch (CourierException e) {
                        String text = (null != _courier) ? "Courier <"
                                + _courier.getClass().getName()
                                + ".deliverAsync(Message) FAILED"
                                : "NULL courier can't deliverAsync Message";
                        _logger.error(text, e);
	                    rollbackJMSTransaction();
                        continue;
                    }
                    continue;
                }
                catch (InvocationTargetException e) {
                    _logger.error("Problems invoking method <"
                            + _processMethod.getName() + ">", e);
                    rollbackJMSTransaction();
                }
                catch (IllegalAccessException e) {
                    _logger.error("Problems invoking method <"
                            + _processMethod.getName() + ">", e);
                    rollbackJMSTransaction();
                }
                catch (Exception e) {
                    _logger.error("Unexpected problem", e);
                    rollbackJMSTransaction();
                }
            }
        }

        _logger.debug("run() method of " + this.getClass().getSimpleName()
                + " finished on thread " + Thread.currentThread().getName());
    } // ________________________________

    /**
     * Handle the threaded destroy of the managed instance.
     *
     * @throws ManagedLifecycleException for errors while destroying.
     */
    protected void doThreadedDestroy() throws ManagedLifecycleException {
        cleanup();
    }
    
    private void rollbackJMSTransaction() 
    {
        try
		{
			if ( jmsSession != null && jmsSession.getTransacted() )
				jmsSession.rollback();
		} catch (JMSException e) {
			final String errorMsg = "JMSException during jmsSession.rollback()";
			_logger.error( errorMsg, e );
		}
    }
    

    private void cleanup() {
        try {
            if (_serviceName != null) {
                RegistryUtil.unregister(_serviceCategory, _serviceName, _myEpr);
            }
            if ( jmsSession != null && jmsSession.getTransacted() )
            	jmsSession.rollback();
        } 
        catch (JMSException e)
		{
			e.printStackTrace();
		} finally {
            try {
                if (jmsMessageConsumer != null) {
                    try {
                        jmsMessageConsumer.close();
                    }
                    catch (final JMSException jmse) {
                    } // ignore
                }
            } finally {
                if (jmsSession != null) {
                    jmsConnectionPool.closeSession(jmsSession);
                }
            }
        }
    }

    /**
     * Check for mandatory and optional attributes in parameter tree
     *
     * @throws ConfigurationException -
     *                                if mandatory atts are not right or actionClass not in
     *                                classpath
     */
    protected void checkMyParms() throws ConfigurationException {

        _targetServiceCategory = ListenerUtil.getValue(_config,
                ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG, null);
        _targetServiceName = ListenerUtil.getValue(_config,
                ListenerTagNames.TARGET_SERVICE_NAME_TAG, null);

        if (_targetServiceCategory == null)
        	throw new ConfigurationException("No service category defined!");
        
        if (_targetServiceName == null)
        	throw new ConfigurationException("No service name defined!");
        
        jmsDestinationName = ListenerUtil.getValue(_config,
                JMSEpr.DESTINATION_NAME_TAG, null);

        if (jmsDestinationName == null)
        	throw new ConfigurationException("No queue name defined!");
        
        resolveComposerClass();

        // No problem if selector is null - everything in queue will be returned
        _messageSelector = _config.getAttribute(JMSEpr.MESSAGE_SELECTOR_TAG);
        _logger.debug("No value specified for: " + JMSEpr.MESSAGE_SELECTOR_TAG
                + " - All messages in queue will be received by this listener");
    } // ________________________________

    protected void resolveComposerClass() throws ConfigurationException {
        try {
            String sProcessMethod = null;
            _composerName = _config
                    .getAttribute(ListenerTagNames.GATEWAY_COMPOSER_CLASS_TAG);
            if (null != _composerName) { // class attribute
                _composerClass = ClassUtil.forName(_composerName, getClass());
                Constructor oConst = _composerClass.getConstructor(new Class[]
                        {ConfigTree.class});
                _composer = oConst.newInstance(_config);
                sProcessMethod = _config
                        .getAttribute(
                                ListenerTagNames.GATEWAY_COMPOSER_METHOD_TAG,
                                "process");
            } else {
                _composerName = PackageJmsMessageContents.class.getName();
                _composerClass = PackageJmsMessageContents.class;
                _composer = new PackageJmsMessageContents(PackageJmsMessageContents.createPayloadProxy(_config));
                sProcessMethod = "process";
                _logger
                        .debug("No <" + ListenerTagNames.ACTION_ELEMENT_TAG
                                + "> element found in configuration"
                                + " -  Using default composer class : "
                                + _composerName);
            }

            _processMethod = _composerClass.getMethod(sProcessMethod,
                    new Class[]
                            {Object.class});
        }
        catch (Exception ex) {
            throw new ConfigurationException(ex);
        }
    } // ________________________________

    private void prepareMessageReceiver() throws ConfigurationException,
            JMSException, ConnectionException, NamingContextException {
        jmsSession = null;
        jmsDestination = null;

        Properties environment = new Properties();

        String sJndiURL = _config.getAttribute(JMSEpr.JNDI_URL_TAG);
        String sJndiContextFactory = _config
                .getAttribute(JMSEpr.JNDI_CONTEXT_FACTORY_TAG);
        String sJndiPkgPrefix = _config
                .getAttribute(JMSEpr.JNDI_PKG_PREFIX_TAG);
        if (sJndiURL != null)
            environment.setProperty(Context.PROVIDER_URL, sJndiURL);
        if (sJndiContextFactory != null)
            environment.setProperty(Context.INITIAL_CONTEXT_FACTORY,
                    sJndiContextFactory);
        if (sJndiPkgPrefix != null)
            environment.setProperty(Context.URL_PKG_PREFIXES, sJndiPkgPrefix);
        Set<String> names = _config.getAttributeNames();
        for (String name : names) {
            if (name.startsWith("java.naming.")) {
                environment.setProperty(name, _config.getAttribute(name));
            }
        }
        Context oJndiCtx = NamingContextPool.getNamingContext(environment);
        try {
            String sFactClass = ListenerUtil.getValue(_config,
                    JMSEpr.CONNECTION_FACTORY_TAG, "ConnectionFactory");
            if (null == _config.getAttribute(JMSEpr.CONNECTION_FACTORY_TAG))
                _logger.debug("No value specified for "
                        + JMSEpr.CONNECTION_FACTORY_TAG + " attribute"
                        + " -  Using default of: '" + sFactClass + "'");
            _serviceCategory = _config
                    .getAttribute(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG);
            _serviceName = _config.getAttribute(ListenerTagNames.SERVICE_NAME_TAG);
    
            String destType = _config.getAttribute(JMSEpr.DESTINATION_TYPE_TAG);
            boolean persistent = Boolean.valueOf( _config.getAttribute(JMSEpr.PERSISTENT_TAG));
            boolean transacted = Boolean.valueOf( _config.getAttribute(JMSEpr.TRANSACTED_TAG));
            
            String acknowledgeMode = _config.getAttribute(JMSEpr.ACKNOWLEDGE_MODE_TAG);
            
            final String username =  _config.getAttribute( JMSEpr.JMS_SECURITY_PRINCIPAL_TAG );
            final String password =  _config.getAttribute( JMSEpr.JMS_SECURITY_CREDENTIAL_TAG );
            if ( username != null && password != null )
            {
    	        environment.put( JMSEpr.JMS_SECURITY_PRINCIPAL_TAG, username );
    	        environment.put( JMSEpr.JMS_SECURITY_CREDENTIAL_TAG, password );
            }
            _logger.debug( "JMSGateway isTransacted = " + transacted );
            
            _myEpr = (null == _serviceName) ? null : new JMSEpr(JMSEpr.ONE_ONE_PROTOCOL, destType,
                jmsDestinationName, sFactClass, environment, _messageSelector, persistent, acknowledgeMode,
                username, password, transacted );
            jmsConnectionPool = JmsConnectionPoolContainer.getPool(environment, sFactClass, destType,username, password, transacted);
            	
            try {
                jmsSession = _myEpr != null ? jmsConnectionPool.getSession(((JMSEpr)_myEpr).getAcknowledgeMode()):
                	jmsConnectionPool.getSession(Session.AUTO_ACKNOWLEDGE);
            		
            }
            catch (NamingException ne) {
                throw new ConfigurationException("Failed to obtain queue session from pool", ne);
            }
    
            try {
                jmsDestination = (Destination) oJndiCtx.lookup(jmsDestinationName);
            }
            catch (NamingException nex) {
                try {
                    oJndiCtx = NamingContextPool.replaceNamingContext(oJndiCtx, environment);
                    jmsDestination = (Destination) oJndiCtx.lookup(jmsDestinationName);
                }
                catch (NamingException ne) {
                    if(jmsSession instanceof QueueSession) {
                        jmsDestination = jmsSession.createQueue(jmsDestinationName);
                    } else {
                        jmsDestination = jmsSession.createTopic(jmsDestinationName);
                    }
                }
            }
        } finally {
            NamingContextPool.releaseNamingContext(oJndiCtx) ;
        }

        if(jmsSession instanceof QueueSession && jmsDestination instanceof Queue) {
            jmsMessageConsumer = ((QueueSession)jmsSession).createReceiver((Queue)jmsDestination, _messageSelector);
        } else if(jmsSession instanceof TopicSession && jmsDestination instanceof Topic) {
            jmsMessageConsumer = ((TopicSession)jmsSession).createSubscriber((Topic)jmsDestination, _messageSelector, false);
        } else {
            try {
                throw new ConfigurationException("The JMS destination identified by name '" + jmsDestinationName + "' must match it's configured destination type '" + jmsDestinationName + "'.");
            } finally {
                cleanup();
            }
        }
    } // ________________________________

    /**
     * Receive one message and retry if connection
     *
     * @return javax.jms.Message - One input message, or null
     */
    protected javax.jms.Message receiveOne() {
        while (isRunning())
            try {
                javax.jms.Message ret = jmsMessageConsumer.receive(200);
                if (null != ret)
                    return ret;
            }
            catch (JMSException oJ) {
                if (_logger.isDebugEnabled()) {
                    _logger
                            .debug(
                                    "JMS error on receive.  Attempting JMS Destination reconnect.",
                                    oJ);
                }
                try {
                    prepareMessageReceiver();
                    errorDelay = 0;
                }
                // try to reconnect to the queue
                catch (Exception e) {
                    if (_logger.isDebugEnabled()) {
                        _logger.debug("Reconnecting to Queue", e);
                    }
                    if (errorDelay == 0) {
                        errorDelay = MIN_ERROR_DELAY;
                    } else if (errorDelay < MAX_ERROR_DELAY) {
                        errorDelay <<= 1;
                    }
                    _logger
                            .warn("Error reconnecting to Queue, backing off for "
                                    + errorDelay + " milliseconds");
                    waitForRunningStateChange(
                            ManagedLifecycleThreadState.STOPPING, errorDelay);
                }
            }
        return null;
    } // ________________________________

    protected final static Logger _logger = Logger
            .getLogger(JmsGatewayListener.class);

    protected String jmsDestinationName;

    protected Session jmsSession;

    protected Destination jmsDestination;

    protected MessageConsumer jmsMessageConsumer;

    protected String _messageSelector;

    protected ConfigTree _config;

    protected String _serviceCategory, _serviceName;

    protected String _targetServiceCategory, _targetServiceName;

    protected EPR _myEpr;

    protected Collection<EPR> _targetEprs;

    protected String _composerName;

    protected Class _composerClass;

    protected Object _composer;

    protected Method _processMethod;

    protected Courier _courier;

    protected JmsConnectionPool jmsConnectionPool;

    /**
     * The minimum error delay.
     */
    private static final long MIN_ERROR_DELAY = 1000;

    /**
     * The maximum error delay.
     */
    private static final long MAX_ERROR_DELAY = (MIN_ERROR_DELAY << 5);

    /**
     * The error delay.
     */
    private long errorDelay;

}
