/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */

package org.jboss.soa.esb.listeners.message;

import java.sql.Timestamp;
import java.util.Observable;
import java.util.Observer;

import org.jboss.system.ServiceMBeanSupport;

/**
 * MessageCounter is a MBean implementation which keeps track of message data and 
 * metadata.
 * 
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a> 
 *
 */
public class MessageCounter extends ServiceMBeanSupport implements MessageCounterMBean, Observer {

	private String lastSuccessfulMessageDate;
	private String lastFailedMessageDate;
	private int failedMessageCount;
	private int successMessageCount;
	private long totalProcessTime;
	
	/**
	 * Constructor.
	 */
	public MessageCounter () {
		DeliveryObservableLogger.getInstance().addObserver(this);
		resetCounts();
	}
	
	/**
	 * Get the total number of failed messages.
	 * @return total number of failed messages
	 */
	public int getFailedMessageCount() {
		return failedMessageCount;
	}

	/**
	 * Get the time the last message was processed at.
	 * @return time the last message was processed at
	 */
	public String getLastSuccessfulMessageDate() {
		return lastSuccessfulMessageDate;
	}

	/**
	 * Get the time the last message was processed at.
	 * @return time the last message was processed at
	 */
	public String getLastFailedMessageDate() {
		return lastFailedMessageDate;
	}
	
	/**
	 * Get the total number of processed messages, both successful and failed.
	 * @return total number of processed messages
	 */
	public int getTotalMessageCount() {
		return (failedMessageCount + successMessageCount);
	}

	/**
	 * Get the total number of successfully processed messages.
	 * @return total number of successfully processed messages
	 */
	public int getSuccessfulMessageCount() {
		return successMessageCount;
	}

	/**
	 * Update the message counter based on the MessageStatusBean that is returned
	 * from the observable.
	 * @param o the observable object
	 * @param arg the MessageStatusBean 
	 */
	public void update(Observable o, Object arg) {
		MessageStatusBean msb = (MessageStatusBean) arg;
		
		if (msb.getMessageStatus().equals(MessageStatusBean.MESSAGE_SENT)) {
			successMessageCount++;
			Timestamp ts = new Timestamp(msb.getMessageTime());
			lastSuccessfulMessageDate = ts.toString();
		} else if (msb.getMessageStatus().equals(MessageStatusBean.MESSAGE_FAILED)) {
			failedMessageCount++;
			Timestamp ts = new Timestamp(msb.getMessageTime());
			lastFailedMessageDate = ts.toString();
		}
		
		totalProcessTime += msb.getProcessTime();
    }

	/**
	 * Get the average time to process a message.
	 * @return average time to process a message 
	 */
	public Double getAverageSuccessTime() {
		if ((successMessageCount) > 0) {
			double ms = (double) totalProcessTime / (1000000 * successMessageCount);
			return new Double(ms);
		} else {
			return null;
		}
	}
	
	/**
	 * Reset the counts - this resets the totals and the last message time. 
	 */
	public void resetCounts() {
		lastSuccessfulMessageDate = "---";
		lastFailedMessageDate = "---";
		successMessageCount = 0;
		failedMessageCount = 0;
		totalProcessTime = 0;
	}
}
