/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners.message;

import java.util.Hashtable;
import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.DynamicMBean;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;

/**
 * Service message counter is a MBean that displays an action-by-action breakdown of
 * how many failed and successful messages have been processed and shows the processing time
 * of each.
 * 
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 * @since Version 4.2
 */
public class ServiceMessageCounter implements DynamicMBean {
	private Hashtable<String, Integer> actionCounterHash;
	private Hashtable<String, Integer> actionFailedCounterHash;
	private Hashtable<String, Long> actionProcessTimeHash;
	private ConfigTree m_config;
	private ConfigTree[] actionArray;
	private Integer serviceCount;
	
	public static final String RESET_COUNTER = "resetCounter";
	private static final String MESSAGE_COUNTER = "messages successfully processed count";
	private static final String FAILED_MESSAGE_COUNTER = "messages failed count";
	private static final String PROCESSING_TIME = "processing time";
	private static final String OVERALL_SERVICE_COUNT = "overall service message count";
	
	/**
	 * Constructor
	 * @param f_config config tree
	 */
	public ServiceMessageCounter(ConfigTree f_config) {
		actionCounterHash = new Hashtable<String, Integer>();
		actionFailedCounterHash = new Hashtable<String, Integer>();
		actionProcessTimeHash = new Hashtable<String, Long>();
		serviceCount = new Integer(0);
		
		m_config = f_config;
		actionArray = m_config.getChildren(ListenerTagNames.ACTION_ELEMENT_TAG);

		initHashes();
	}
	
	/**
	 * Return an action id.   If there's an action name, use that, if not,
	 * use the "action" attribute on the action.
	 * @param ct config tree
	 * @return action id
	 */
	public String getActionId(ConfigTree ct) {
		if (ct.getAttribute("name") != null) {
			return ct.getAttribute("name");
		} else if (ct.getAttribute("action") != null) {
			return ct.getAttribute("action");
		}
		return null;
	}
	
	/**
	 * Increment the total message count of this service.
	 */
	public void incrementTotalCount() {
		serviceCount = new Integer(serviceCount.intValue() + 1);
	}
	
	/**
	 * Initialize the hashes by setting the counts and the processing time to an initial value of 0. 
	 */
	public void initHashes() {
		ConfigTree[] actionList = m_config.getChildren(ListenerTagNames.ACTION_ELEMENT_TAG);
		
		for (ConfigTree actionConfig : actionList) {
			String actionId = getActionId(actionConfig);
		 	actionCounterHash.put(actionId + " " + MESSAGE_COUNTER, new Integer(0));
			actionFailedCounterHash.put(actionId + " " + FAILED_MESSAGE_COUNTER, new Integer(0));
			actionProcessTimeHash.put(actionId + " " + PROCESSING_TIME, new Long(0));
		}		
		serviceCount = new Integer(0);
	}
	
	/**
	 * Reset the counters - set all the entries in the action counter hash 
	 * and in the action process time hash to zero.
	 */
	public void resetCounter() {
		serviceCount = new Integer(0);
		
		for (String key : actionCounterHash.keySet()) {
			actionCounterHash.put(key, new Integer(0));
		}
		
		for (String key : actionFailedCounterHash.keySet()) {
			actionFailedCounterHash.put(key, new Integer(0));
		}
		
		for (String key : actionProcessTimeHash.keySet()) {
			actionProcessTimeHash.put(key, new Long(0));
		}
	}
	
	/**
	 *  This creates the MBeanInfo object provided.     We are returning generic 
	 *  text for the attribute descriptions (the word Property and the name of the 
	 *  attribute), all of the attributes are read-only, and we provide four 
	 *  invocation methods - start/stop/initialise/destroy on the Lifecycle. 
	 */
    public MBeanInfo getMBeanInfo() {
		
		int count = actionCounterHash.size() + actionProcessTimeHash.size()
			+ actionFailedCounterHash.size() + 1;
        MBeanAttributeInfo[] attrs = new MBeanAttributeInfo[count];
        int counter = 0;
              
		for (String key : actionCounterHash.keySet()) {
            attrs[counter] = new MBeanAttributeInfo(
                    key, "java.lang.Integer", "Property " + key, true, false, false);
            counter++;
		}
		
		for (String key : actionProcessTimeHash.keySet()) {
            attrs[counter] = new MBeanAttributeInfo(
                    key, "java.lang.Double", "Property " + key, true, false, false);
            counter++;
		}
		
		for (String key : actionFailedCounterHash.keySet()) {
            attrs[counter] = new MBeanAttributeInfo(
                    key, "java.lang.Integer", "Property " + key, true, false, false);
            counter++;
		}
		
		MBeanAttributeInfo overallCount = new MBeanAttributeInfo(OVERALL_SERVICE_COUNT, "java.lang.Integer",
				"Property " + OVERALL_SERVICE_COUNT, true, false, false);
		attrs[counter] = overallCount;
		counter++;
			
        MBeanOperationInfo[] opers = {
        	new MBeanOperationInfo(
        			RESET_COUNTER, "Reset the counter",
                	null, "void", MBeanOperationInfo.ACTION)
        };
        return new MBeanInfo(
                this.getClass().getName(), "Service Message Counter MBean",
                attrs, null, opers, null); // notifications
	}

	/* (non-Javadoc)
	 * @see javax.management.DynamicMBean#getAttribute(java.lang.String)
	 */
	public Object getAttribute(String key) throws AttributeNotFoundException, MBeanException, ReflectionException {
    	if (actionCounterHash.containsKey(key)) {
    		Integer value = null;
    		value = actionCounterHash.get(key);
    		return value;
    	} else if (actionProcessTimeHash.containsKey(key)) {
    		Long processTotal = actionProcessTimeHash.get(key);
    		String actionId = key.substring(0, key.indexOf(PROCESSING_TIME)-1);
    		Integer successCount = actionCounterHash.get(actionId + " " + MESSAGE_COUNTER);
    		Double value = null;
    		
    		if (successCount.intValue() > 0) {
    			value = ((double) processTotal / successCount.intValue());
    		} else {
    			value = null;
    		}
    		return value;
    	} else if (actionFailedCounterHash.containsKey(key)) {
    		Integer value = actionFailedCounterHash.get(key);
    		return value;
    	} else if (OVERALL_SERVICE_COUNT.equals(key)) {
    		return serviceCount;
    	}
    	return null;
    }

	/* (non-Javadoc)
	 * @see javax.management.DynamicMBean#getAttributes(java.lang.String[])
	 */
	public AttributeList getAttributes(String[] arg0) {
		AttributeList attributeList = new AttributeList();
		for (String key : actionCounterHash.keySet()) {
			Attribute at = new Attribute(key, actionCounterHash.get(key).toString());
			attributeList.add(at);
		}
		
		Attribute overallCount = new Attribute(OVERALL_SERVICE_COUNT, serviceCount);
		attributeList.add(overallCount);
		
		for (String key : actionProcessTimeHash.keySet()) {
			Long processTotal = actionProcessTimeHash.get(key);
			String actionId = key.substring(0, key.indexOf(PROCESSING_TIME));
			Integer successCount = actionCounterHash.get(actionId + " " + MESSAGE_COUNTER);
			String avgTime = null;
			if (successCount.intValue() > 0) {
				avgTime = ((double) processTotal / successCount.intValue()) + " ns";
			}
			Attribute at = new Attribute(key, avgTime);
			attributeList.add(at);
		}
		
		for (String key : actionFailedCounterHash.keySet()) {
			Attribute at = new Attribute(key, actionFailedCounterHash.get(key).toString());
			attributeList.add(at);
		}

		return attributeList;
	}

	/* (non-Javadoc)
	 * @see javax.management.DynamicMBean#invoke(java.lang.String, java.lang.Object[], java.lang.String[])
	 */
	public Object invoke(String method, Object[] arg1, String[] arg2) throws MBeanException, ReflectionException {
		if (method.equalsIgnoreCase(RESET_COUNTER)) {
				resetCounter();
			return "Invoking the " + method + " on the lifecycle.";
		} else {
			throw new ReflectionException(new NoSuchMethodException(method));
		}
	}

	/* (non-Javadoc)
	 * @see javax.management.DynamicMBean#setAttribute(javax.management.Attribute)
	 */
	public void setAttribute(Attribute arg0) throws AttributeNotFoundException, InvalidAttributeValueException, MBeanException, ReflectionException {
	}

	/* (non-Javadoc)
	 * @see javax.management.DynamicMBean#setAttributes(javax.management.AttributeList)
	 */
	public AttributeList setAttributes(AttributeList arg0) {
		return null;
	}

	/**
	 * Register this MBean with JBoss.
	 */
	protected void registerMBean() {
		MBeanServer mbeanServer = null;
		try {
			mbeanServer = MBeanServerLocator.locateJBoss();
		} catch (IllegalStateException ise) {
			// If we can't find a JBoss MBeanServer, just return
			// Needed for unit tests
			return;
		}
		
		ObjectName listObjectName = getObjectName();
		
		if (mbeanServer.isRegistered(listObjectName)) {
	    	try {
	    		mbeanServer.unregisterMBean(listObjectName);
			} catch (InstanceNotFoundException e) {
				e.printStackTrace();
			} catch (MBeanRegistrationException e) {
				e.printStackTrace();
			}
        }
		
	    try {
	    	mbeanServer.registerMBean(this, listObjectName);
		} catch (InstanceAlreadyExistsException e) {
			e.printStackTrace();
		} catch (MBeanRegistrationException e) {
			e.printStackTrace();
		} catch (NotCompliantMBeanException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Update the ServiceMessageCounter
	 * @param asb ActionStatusBean
	 */
	public void update(ActionStatusBean asb) {		
		String actionName = getActionId(actionArray[asb.getProcCount()]); 

		if (ActionStatusBean.ACTION_SENT.equals(asb.getStatus())) {
			Integer count = actionCounterHash.get(actionName + " " + MESSAGE_COUNTER);
			count = count.intValue() + 1;
			actionCounterHash.put(actionName + " " + MESSAGE_COUNTER, count);
			Long time = actionProcessTimeHash.get(actionName + " " + PROCESSING_TIME);
			time = time.longValue() + asb.getProcTime();
			actionProcessTimeHash.put(actionName + " " + PROCESSING_TIME, time);
		} else if (ActionStatusBean.ACTION_FAILED.equals(asb.getStatus())) {
			Integer count = actionFailedCounterHash.get(actionName + " " + FAILED_MESSAGE_COUNTER);
			count = count.intValue() + 1;
			actionFailedCounterHash.put(actionName + " " + FAILED_MESSAGE_COUNTER, count);
		}		
	}
	
	protected ObjectName getObjectName()
	{
		ObjectName listObjectName = null;
		try {
			String deploymentName = m_config.getParent().getAttribute(ListenerTagNames.DEPLOYMENT_NAME_TAG);
			String serviceName = m_config.getAttribute(ListenerTagNames.SERVICE_NAME_TAG);

			StringBuffer objectName = new StringBuffer("category=MessageCounter");
			if (deploymentName != null) {
				objectName.append(",").append(ListenerTagNames.DEPLOYMENT_NAME_TAG).append("=").append(deploymentName);
			}
			
			if (serviceName != null) {
				if (objectName.length() > 0) {
					objectName.append(",");
				}
				objectName.append(ListenerTagNames.SERVICE_NAME_TAG).append("=").append(serviceName);
			}
						
			listObjectName = new ObjectName("jboss.esb:" + objectName.toString());
		} catch (MalformedObjectNameException e1) {
			e1.printStackTrace();
		} catch (NullPointerException e1) {
			e1.printStackTrace();
		}
		return listObjectName;
	}
}
