/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.notification;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;

/**
 * Holds lists of NotificationTarget objects so that the NotificationHandler EJB
 * behaviour can be controlled at runtime by client applications
 * <p>
 * Description: Objects of this class will hold a list of objects that extend
 * the NotificationTarget base class. It's main purpose is to allow Rosetta
 * users to configure lists of recipients of specific events, and request the
 * NotificationHandler EJB to take care of triggering the events to each object
 * specified as a child "target" element
 * </p>
 * <p>
 * Author: Heuristica - Buenos Aires - Argentina
 * </p>
 * 
 * @version 1.0
 */
public class NotificationList extends ConfigTree
{
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Mnemonic name for the XML element name for the serialized form of objects
	 * of this class
	 */
	public static final String ELEMENT = "NotificationList";

	public static final String TYPE = "type";

	public static final String CHILD_TGT = "target";

	private String m_sType;

	protected static Logger _logger = Logger.getLogger(NotificationList.class);

	/**
	 * Instantiate an object according to the contents of &lt;arg 1&gt;
	 * 
	 * @param p_oP
	 *            ConfigTree - Parameter object containing the information
	 *            needed to instantiate this object, including child elements
	 *            named "target" that represent each individual
	 *            NotificationTarget object
	 */
	public NotificationList (ConfigTree p_oP)
	{
		super(p_oP);
		m_sType = p_oP.getAttribute(TYPE);
		if (null != m_sType) m_sType = m_sType.toLowerCase();
	} // __________________________________

	private NotificationTarget[] getTargets () throws ConfigurationException
	{
		ConfigTree[] oaTgts = super.getChildren(CHILD_TGT);
		NotificationTarget[] oaRet = new NotificationTarget[oaTgts.length];

		for (int i1 = 0; i1 < oaRet.length; i1++)
		{
			oaRet[i1] = NotificationTarget.fromParams(oaTgts[i1]);
		}

		return oaRet;
	} // __________________________________

	/**
	 * Invoke the sendNotification() method for all targets in the list. The
	 * method iterates along it's NotificationTarget child nodes and invokes the
	 * sendNotification() method with the same object to all of them
	 * 
	 * @param message
	 *            Object - The event to notify to all targets
	 * @throws NotificationException -
	 *             use Exception.getMessage() at runtime
	 */
	public void sendNotification (Message message) throws NotificationException
	{
		try
		{
			NotificationTarget[] oaTgt = getTargets();
	
			for (int i1 = 0; i1 < oaTgt.length; i1++)
			{
				oaTgt[i1].sendNotification(message);
			}
		}
		catch (ConfigurationException ex)
		{
			throw new NotificationException(ex);
		}
	} // __________________________________

	/**
	 * Is this an OK (success) NotificationList ?
	 * 
	 * @return boolean - Yes, Attribute "type" of constructor parameters starts
	 *         with "ok", OR there was no "type" attribute in the constructor
	 *         <li>A NotificationList can be both OK and Error if no "type"
	 *         attribute was specified at constructor time in the ConfigTree
	 *         argument</li>
	 * @see ConfigTree#getAttribute(String)
	 */
	public boolean isOK ()
	{
		return (null == m_sType) ? true : m_sType.startsWith("ok");
	}

	/**
	 * Is this an Error (failure) NotificationList ?
	 * 
	 * @return boolean - Yes, Attribute "type" of constructor parameters starts
	 *         with "err", OR there was no "type" attribute in the constructor
	 *         <li>A NotificationList can be both OK and Error if no "type"
	 *         attribute was specified at constructor time in the ConfigTree
	 *         argument</li>
	 * @see ConfigTree#getAttribute(String)
	 */
	public boolean isErr ()
	{
		return (null == m_sType) ? true : m_sType.startsWith("err");
	}

	public static void notifyAll (ConfigTree[] list, Message message)
	{
		for (ConfigTree tree : list)
		{
			ConfigTree[] targets = tree.getChildren(NotificationList.CHILD_TGT);
			for (ConfigTree curr : targets)
			{
				try
				{
					NotificationTarget target = NotificationTarget
							.fromParams(curr);
					target.sendNotification(message);
				}
				catch (Exception e)
				{
					_logger.error(
							"Can't instantiate target " + curr.toString(), e);
				}
			}
		}
	} // ________________________________

} // ____________________________________________________________________________
