/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.notification;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.nio.ByteBuffer;

import javax.jms.DeliveryMode;
import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.ObjectMessage;
import javax.jms.TextMessage;
import javax.jms.TopicConnection;
import javax.naming.Context;
import javax.naming.NamingException;

import junit.framework.TestCase;

import org.jboss.internal.soa.esb.rosetta.pooling.ConnectionException;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.NamingContextException;
import org.jboss.soa.esb.helpers.NamingContextPool;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.mockejb.jms.MockTopic;
import org.mockejb.jms.TopicConnectionFactoryImpl;
import org.mockejb.jndi.MockContextFactory;

/**
 * NotifyTopics unit tests.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class NotifyTopicsUnitTest extends TestCase {

	private MockTopic mockTopic1;
	private MockTopic mockTopic2;
	private NotifyTopics notifyTopics;
	
	protected void setUp() throws Exception {
		MockContextFactory.setAsInitial();
                final Context ctx = NamingContextPool.getNamingContext(null);
                try
                {
                    ctx.rebind(NotifyTopics.CONNECTION_FACTORY, new MockTopicConnectionFactory());
                }
                finally
                {
                    NamingContextPool.releaseNamingContext(ctx) ;
                }
		ConfigTree rootEl = new ConfigTree("rootEl");

		addMessagePropertyConfigs(rootEl);
		addTopicConfig(rootEl, "topic1");
		addTopicConfig(rootEl, "topic2");
		mockTopic1 = createAndBindTopic("topic1");
		mockTopic2 = createAndBindTopic("topic2");
		
		notifyTopics = new NotifyTopics(rootEl);
	}

	protected void tearDown() throws Exception {
		MockContextFactory.revertSetAsInitial();		
	}

	public void test_StringObj() throws Exception {
        org.jboss.soa.esb.message.Message message = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
        message.getBody().add("Hello".getBytes());
		notifyTopics.sendNotification(message);
		
		checkTopicTextMessage(mockTopic1, 0, "Hello");
		checkTopicTextMessage(mockTopic2, 0, "Hello");
	}
	
	public void test_NonStringObj() throws Exception {
        org.jboss.soa.esb.message.Message message = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
        message.getBody().add((new Integer(123).toString().getBytes()));
		notifyTopics.sendNotification(message);
		
		checkTopicObjectMessage(mockTopic1, 0, new Integer(123).toString().getBytes());
		checkTopicObjectMessage(mockTopic2, 0, new Integer(123).toString().getBytes());
	}
	
	public void test_sendNotification_persistent() throws ConfigurationException, JMSException, ConnectionException 
	{
		assertEquals( DeliveryMode.PERSISTENT, notifyTopics.deliveryModes[0] );
	}
	
	public void test_sendNotification_priority() throws ConfigurationException, JMSException, ConnectionException 
	{
		assertEquals( Message.DEFAULT_PRIORITY, notifyTopics.priorities[0] );
	}
	
	public void test_sendNotification_ttl() throws ConfigurationException, JMSException, ConnectionException 
	{
		assertEquals( Message.DEFAULT_TIME_TO_LIVE, notifyTopics.timeToLives[0] );
	}
	
	public void test_createException()
	{
		final String msg = "[JMSException while publishing to : /topic/SomeName";
		try
		{
			String string = NotifyUtil.createExceptionErrorString( msg, new JMSException("junit test dummy exception") );
			assertNotNull( string );
		}catch(Exception e )
		{
			fail(e.getMessage());
		}
	}

	private void checkTopicTextMessage(MockTopic mockTopic, int messageIdx, String expectedText) throws JMSException {
		assertTrue(mockTopic.getMessages().size() > messageIdx);		
		Message message = mockTopic.getMessageAt(0);
		assertTrue(message instanceof TextMessage);
		assertEquals(expectedText, ((TextMessage)message).getText());
		assertEquals("testpropvalue", message.getStringProperty("testpropname"));
	}

	private void checkTopicObjectMessage(MockTopic mockTopic, int messageIdx, Object expectedObj) throws JMSException {
		assertTrue(mockTopic.getMessages().size() > messageIdx);		
		Message message = mockTopic.getMessageAt(0);
		assertTrue(message instanceof ObjectMessage);
        ByteBuffer byteBuffer = ByteBuffer.wrap((byte[]) ((ObjectMessage) message).getObject());
        ByteBuffer expectedByteBuffer = ByteBuffer.wrap((byte[]) (expectedObj));
		assertEquals(expectedByteBuffer, byteBuffer);
		
		// Note that the property bindings don't seem to work in this test i.e.
		// it's returning null but should be returning the same as for a 
		// TextMessage (See above).  This is most likely a mockejb lib issue
		// and so we're ignoring it :-)
		assertEquals(null, message.getStringProperty("testpropname"));
	}
	
	private void addMessagePropertyConfigs(ConfigTree rootEl) {
		ConfigTree propEl = new ConfigTree(NotifyTopics.CHILD_MSG_PROP,rootEl);
		
		propEl.setAttribute(NotifyJMS.ATT_PROP_NAME, "testpropname");
		propEl.setAttribute(NotifyJMS.ATT_PROP_VALUE, "testpropvalue");
	}

	private void addTopicConfig(ConfigTree rootEl, String topicName) {
		ConfigTree topicEl = new ConfigTree(NotifyTopics.CHILD_TOPIC,rootEl);
		
		topicEl.setAttribute(NotifyJMS.ATT_DEST_NAME, topicName);
	}

	private MockTopic createAndBindTopic(String topicName) throws NamingException, NamingContextException {
		MockTopic mockTopic = new MockTopic(topicName);
		
		final Context ctx = NamingContextPool.getNamingContext(null) ;
		try
		{
		    ctx.rebind(topicName, mockTopic);
		}
		finally
		{
		    NamingContextPool.releaseNamingContext(ctx) ;
		}
		return mockTopic;
	}
        
        private static final class MockTopicConnectionFactory extends TopicConnectionFactoryImpl
        {
            @Override
            public TopicConnection createTopicConnection() throws JMSException
            {
                return (TopicConnection)Proxy.newProxyInstance(getClass().getClassLoader(), new Class[] {TopicConnection.class},
                        new MockTopicExceptionHandlerInvocationHandler(super.createTopicConnection())) ;
            }
        }
        
        private static final class MockTopicExceptionHandlerInvocationHandler implements InvocationHandler
        {
            private final TopicConnection topicConnection ;
            private ExceptionListener exceptionListener ;
            
            MockTopicExceptionHandlerInvocationHandler(final TopicConnection topicConnection)
            {
                this.topicConnection = topicConnection ;
            }
            
            public Object invoke(final Object proxy, final Method method, final Object[] args)
                throws Throwable
            {
                final String methodName = method.getName() ;
                if ("setExceptionListener".equals(methodName))
                {
                    exceptionListener = (ExceptionListener)args[0] ;
                    return null ;
                }
                else if ("getExceptionListener".equals(methodName))
                {
                    return exceptionListener ;
                }
                else
                {
                    return method.invoke(topicConnection, args) ;
                }
            }
        }
}
