/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.routing.cbr;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.log4j.Logger;
import org.apache.log4j.Level;
import org.drools.RuleBase;
import org.drools.RuleBaseFactory;
import org.drools.StatefulSession;
import org.drools.compiler.DroolsParserException;
import org.drools.compiler.PackageBuilder;
import org.drools.compiler.PackageBuilderConfiguration;
import org.drools.rule.Package;
import org.jboss.internal.soa.esb.services.routing.cbr.JBRulesCounter;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.lifecycle.LifecyclePriorities;
import org.jboss.soa.esb.lifecycle.LifecycleResource;
import org.jboss.soa.esb.lifecycle.LifecycleResourceException;
import org.jboss.soa.esb.lifecycle.LifecycleResourceFactory;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.routing.MessageRouterException;
import org.jboss.soa.esb.services.routing.cbr.CBRException;
import org.jboss.soa.esb.services.routing.cbr.ContentBasedRouter;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * The Implementation of a rule based Content Router. Here we use JBossRules. We
 * keep a HashMap of so called working memories for performance reasons.
 * 
 * @author kstam@redhat.com
 * 
 */
public class JBossRulesRouter implements ContentBasedRouter
{
        /**
         * The lifecycle resource factory.
         */
        private static final LifecycleResourceFactory<Map<String, RuleBase>> lifecycleRuleBaseFactory = new LifecycleRuleBaseFactory() ;
        /**
         * Lifecycle couriers.
         */
        private static final LifecycleResource<Map<String, RuleBase>> lifecycleRuleBases =
            new LifecycleResource<Map<String,RuleBase>>(lifecycleRuleBaseFactory,
                    LifecyclePriorities.RULE_BASE_PRIORITY) ;
        /**
         * The lifecycle resource factory.
         */
        private static final LifecycleResourceFactory<Map<String, String>> lifecycleRuleSetFactory = new LifecycleRuleSetFactory() ;
        /**
         * Lifecycle couriers.
         */
        private static final LifecycleResource<Map<String, String>> lifecycleRuleSets =
            new LifecycleResource<Map<String,String>>(lifecycleRuleSetFactory,
                    LifecyclePriorities.RULE_BASE_PRIORITY) ;
        
	private static Logger logger = Logger.getLogger(JBossRulesRouter.class);
	
	private JBRulesCounter rulesCounter = null;
	
	/**
     * Route the message with a reference to the ruleSets supplied in the message.
     * Not implemented.
	 * 
	 * @param message - Message that needs routing.
	 */
	public List<String> route (Message message)
    throws MessageRouterException
	{
        logger.error("Not implemented");
		return null;
	}

	/**
	 * Route the message, using the given ruleSet (drl).
	 * 
	 * @param ruleSet -
	 *            Filename of the drl that will be used.
	 * @param message -
	 *            Message that needs routing.
	 */
	public List<String> route (String ruleSet, boolean ruleReload,
			Message message, List<Object> objectList) throws MessageRouterException
	{
		return route(ruleSet, null, ruleReload, message, objectList);
	}

	/**
	 * Route the message, where the routing rules are supplied as part of the
	 * message itself. We don't support this yet, as I don't see the need right
	 * now. However when the need arises, this is where it goes.
	 * 
	 * @param ruleSet -
	 *            Filename of the drl that will be used.
	 * @param ruleLanguage -
	 *            Filename of the dsl that will be used.
	 * @param message -
	 *            Message that needs routing.
	 */
	@SuppressWarnings("unchecked")
	public List<String> route (String ruleSet, String ruleLanguage,
			boolean ruleReload, Message message, List<Object> objectList) throws MessageRouterException
	{
        List<String> destinations = new ArrayList();
        long startTime = System.nanoTime();

        try {
    		String newRuleSet = null;
    		boolean isRulesChanged = false;
    
                final Map<String, String> ruleSets = lifecycleRuleSets.getLifecycleResource() ;
    		if (ruleReload)
    		{
    			String currentRuleSet = ruleSets.get(ruleSet);
    			newRuleSet = getRules(ruleSet, ruleLanguage);
    			if (currentRuleSet == null || !currentRuleSet
    					.equals(newRuleSet))
    			{
    				isRulesChanged = true;
    			}
    		}
    		final Map<String, RuleBase> ruleBases = lifecycleRuleBases.getLifecycleResource() ;
    		RuleBase ruleBase = ruleBases.get(ruleSet);
    		if (ruleBase == null || isRulesChanged)
    		{
    			logger.log(Level.DEBUG,
    					"Reading ruleSet from file=" + ruleSet);
    			ruleBase = readRuleBase(ruleSet, ruleLanguage);
    			if (ruleBase != null) ruleBases.put(ruleSet, ruleBase);
    			if (newRuleSet == null)
    			{
    				newRuleSet = getRules(ruleSet, ruleLanguage);
    			}
    			if (ruleSet != null) ruleSets.put(ruleSet, newRuleSet);
    		}
    		StatefulSession workingMemory = ruleBase.newStatefulSession();
    		logger.log(Level.DEBUG,
    				"Obtained message=" + message + " with ruleSet=" + ruleSet);
    		workingMemory.setGlobal("destinations", destinations);
            if (objectList!=null) {
                for (Object object : objectList) {
                    workingMemory.insert(object);
                }
            }
    		workingMemory.insert(message);
    		logger.log(Level.DEBUG, "Fire the JBossRules Engine");
    		workingMemory.fireAllRules();
    		long procTime = System.nanoTime() - startTime;
    		if (rulesCounter != null) {
    			rulesCounter.update(procTime, ruleSet, JBRulesCounter.RULES_SUCCEED);
    		}
    			
    		logger.log(Level.DEBUG,
    				"Outgoing Destinations: " + destinations);
            workingMemory.dispose();
            return destinations;
        } catch (final LifecycleResourceException lre) {
            if (rulesCounter != null) {
                    long procTime = System.nanoTime() - startTime;
                    rulesCounter.update(procTime, ruleSet, JBRulesCounter.RULES_FAILED);
            }
            throw new MessageRouterException("Could not load lifecycle data. " + lre.getMessage(), lre);
        } catch (IOException ioe) {
        	if (rulesCounter != null) {
        		long procTime = System.nanoTime() - startTime;
        		rulesCounter.update(procTime, ruleSet, JBRulesCounter.RULES_FAILED);
        	}
        	throw new MessageRouterException("Could not read the rules. " +  ioe.getMessage(), ioe);
        } catch (DroolsParserException dpe) {
        	if (rulesCounter != null) {
        		long procTime = System.nanoTime() - startTime;
        		rulesCounter.update(procTime, ruleSet, JBRulesCounter.RULES_FAILED);
        	}
        	throw new MessageRouterException("Could not parse the rules. " +  dpe.getMessage(), dpe);
        } catch (CBRException cbre) {
        	if (rulesCounter != null) {
        		long procTime = System.nanoTime() - startTime;
        		rulesCounter.update(procTime, ruleSet, JBRulesCounter.RULES_FAILED);
        	}
        	throw new MessageRouterException("Could not parse the rules. " +  cbre.getMessage(), cbre);
        }
	}

	/**
	 * Reading the rules and dsl from files. Note that we may want to allow
	 * other ways to set the rule, but for now it's just files.
	 */
	private static RuleBase readRuleBase (String rulesFile, String ruleLanguage)
			throws DroolsParserException, IOException, CBRException
	{
		// read in the rules
		logger.debug("Going to read the rule: " + rulesFile);
		InputStream inputStreamDrl = ClassUtil.getResourceAsStream("/" + rulesFile, JBossRulesRouter.class);
		if (inputStreamDrl == null)
		{
			logger.error("Could not find rulesFile: " + rulesFile);
			return null;
		}
		else
		{
	            PackageBuilderConfiguration pkgBuilderCfg = new PackageBuilderConfiguration();
	            //pkgBuilderCfg.setCompiler(PackageBuilderConfiguration.JANINO);
	            PackageBuilder builder = new PackageBuilder(pkgBuilderCfg);
	            try
	            {
                        Reader rules = new InputStreamReader(inputStreamDrl);
                        // this wil parse and compile in one step
                        if (ruleLanguage == null)
                        {
                                builder.addPackageFromDrl(rules);
                        }
                        else
                        {
                                logger.debug("Going to read the language: " + ruleLanguage);
                                InputStream inputStreamDsl = ClassUtil.getResourceAsStream("/" + ruleLanguage, JBossRulesRouter.class);
                                if (inputStreamDsl == null)
                                {
                                        logger.error("Could not find ruleLanguage: " + rulesFile);
                                }
                                else
                                {
                                    try
                                    {
                                        Reader dsl = new InputStreamReader(inputStreamDsl);
                                        builder.addPackageFromDrl(rules, dsl);
                                    }
                                    finally
                                    {
                                        safeClose(inputStreamDsl);
                                    }
                                }
                        }
	            }
	            finally
	            {
	                safeClose(inputStreamDrl);
	            }
			// get the compiled package (which is serializable)
			Package pkg = builder.getPackage();
			// add the package to a rulebase (deploy the rule package).
			try
			{
				RuleBase ruleBase = RuleBaseFactory.newRuleBase();
				ruleBase.addPackage(pkg);
				return ruleBase;
			} catch (Exception ex) {
				throw new CBRException(ex.getMessage(), ex);
			}
		}
	}

	/**
	 * Reading the rules and dsl from files. Note that we may want to allow
	 * other ways to set the rule, but for now it's just files.
	 */
	private static String getRules (String rulesFile, String ruleLanguage)
			throws IOException
	{
		logger.debug("Going to hash the rule: " + rulesFile);
		InputStream inputStreamDrl = ClassUtil.getResourceAsStream("/" + rulesFile, JBossRulesRouter.class);
		if (inputStreamDrl == null)
		{
			logger.error("Could not find rulesFile: " + rulesFile);
			return null;
		}
		else
		{
		    final String rules ;
		    try
		    {
                        rules = getString(inputStreamDrl);
		    }
		    finally
		    {
		        safeClose(inputStreamDrl);
		    }
                    String language = "";
                    if (ruleLanguage != null)
                    {
                            logger.debug("Going to hash the language: " + ruleLanguage);
                            InputStream inputStreamDsl = ClassUtil.getResourceAsStream("/" + ruleLanguage, JBossRulesRouter.class);
                            if (inputStreamDsl == null)
                            {
                                    logger.error("Could not find language: " + ruleLanguage);
                            }
                            else
                            {
                                try
                                {
                                    language = getString(inputStreamDsl);
                                }
                                finally
                                {
                                    safeClose(inputStreamDsl);
                                }
                            }
                    }
                    return rules + language;
		}

	}

	private static void safeClose(final InputStream is)
	{
	    try
	    {
	        is.close() ;
	    }
	    catch (final Throwable th) {} // ignore
	}
	
	private static String getString (InputStream in) throws IOException
	{
		StringBuffer stringBuffer = new StringBuffer();
		byte[] b = new byte[4096];
		for (int i; (i = in.read(b)) != -1;)
		{
			stringBuffer.append(new String(b, 0, i));
		}
		return stringBuffer.toString();
	}

	public void setConfigTree(ConfigTree configTree) {
		rulesCounter = new JBRulesCounter(configTree);
		rulesCounter.registerMBean();
	}
        
        /**
         * The lifecycle resource factory for rule sets.
         * @author kevin
         */
        public static class LifecycleRuleBaseFactory implements LifecycleResourceFactory<Map<String, RuleBase>>
        {
            /**
             * Create a resource object which will be associated with the specified lifecycle identity.
             * @param lifecycleIdentity The associated lifecycle identity.
             * @return The lifecycle resource
             * @throws LifecycleResourceException for errors during construction.
             */
            public Map<String, RuleBase> createLifecycleResource(final String lifecycleIdentity)
                throws LifecycleResourceException
            {
                return new ConcurrentHashMap<String, RuleBase>() ;
            }
        
            /**
             * Destroy a resource object which is associated with the specified lifecycle identity.
             * @param resource The lifecycle resource.
             * @param lifecycleIdentity The associated lifecycle identity.
             * @return The lifecycle resource.
             * @throws LifecycleResourceException for errors during destroy.
             */
            public void destroyLifecycleResource(final Map<String, RuleBase> resource,
                final String lifecycleIdentity)
                throws LifecycleResourceException
            {
            }
        }
	
        /**
         * The lifecycle resource factory for rule sets.
         * @author kevin
         */
        public static class LifecycleRuleSetFactory implements LifecycleResourceFactory<Map<String, String>>
        {
            /**
             * Create a resource object which will be associated with the specified lifecycle identity.
             * @param lifecycleIdentity The associated lifecycle identity.
             * @return The lifecycle resource
             * @throws LifecycleResourceException for errors during construction.
             */
            public Map<String, String> createLifecycleResource(final String lifecycleIdentity)
                throws LifecycleResourceException
            {
                return new ConcurrentHashMap<String, String>() ;
            }
        
            /**
             * Destroy a resource object which is associated with the specified lifecycle identity.
             * @param resource The lifecycle resource.
             * @param lifecycleIdentity The associated lifecycle identity.
             * @return The lifecycle resource.
             * @throws LifecycleResourceException for errors during destroy.
             */
            public void destroyLifecycleResource(final Map<String, String> resource,
                final String lifecycleIdentity)
                throws LifecycleResourceException
            {
            }
        }
}
