/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.soap;

import org.jboss.internal.soa.esb.publish.Publish;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.ResponseHeader;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.wsf.spi.SPIProvider;
import org.jboss.wsf.spi.SPIProviderResolver;
import org.jboss.wsf.spi.deployment.Endpoint;
import org.jboss.wsf.spi.invocation.InvocationContext;
import org.jboss.wsf.spi.invocation.RequestHandler;
import org.jboss.wsf.spi.management.EndpointRegistry;
import org.jboss.wsf.spi.management.EndpointRegistryFactory;

import javax.management.ObjectName;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.UnsupportedEncodingException;
import java.util.Set;

/**
 * JBoss Webservices SOAP Processor.
 * <p/>
 * This action supports invocation of a JBossWS hosted webservice endpoint through any JBossESB hosted
 * listener.  This means the ESB can be used to expose Webservice endpoints for Services that don't
 * already expose a Webservice endpoint.  You can do this by writing a thin Service Wrapper Webservice
 * (e.g. a JSR 181 implementation) that wraps calls to the target Service (that doesn't have a Webservice endpoint),
 * exposing that Service via endpoints (listeners) running on the ESB.  This also means that these Services
 * are invocable over any transport channel supported by the ESB (http, ftp, jms etc).
 *
 * <h3>"ESB Message Aware" Webservice Endpoints</h3>
 * Note that Webservice endpoints exposed via this action have direct access to the current
 * JBossESB {@link org.jboss.soa.esb.message.Message} instance used to invoke this action's
 * {@link #process(org.jboss.soa.esb.message.Message)} method.  It can access
 * the current {@link org.jboss.soa.esb.message.Message} instance via the {@link #getMessage()} method
 * and can change the {@link org.jboss.soa.esb.message.Message} instance via the
 * {@link #setMessage(org.jboss.soa.esb.message.Message)} method.  This means that Webservice endpoints
 * exposed via this action are "ESB Message Aware".
 *
 * <h3>Webservice Endpoint Deployment</h3>
 * Any JBossWS Webservice endpoint can be exposed via ESB listeners using this action.  That includes endpoints that are deployed
 * from inside (i.e. the Webservice .war is bundled inside the .esb) and outside (e.g. standalone Webservice .war deployments,
 * Webservice .war deployments bundled inside a .ear) a .esb deployment.
 *
 * <div style="margin-left: 20">
 * <h4>JAXB Introductions</h4>
 * The native JBossWS SOAP stack uses JAXB to bind to and from SOAP.  This typically means that an unannotated typeset
 * could not be used to build a JSR 181 endpoint on JBossWS.  To overcome this we use a JBossESB and JBossWS feature
 * called "JAXB Introductions" which basically means you can define an XML configuration to "Introduce" the JAXB Annotations.
 * For more on this, see the section on this action in the Message Action Guide.
 * </div>
 *
 * <h3>Action Configuration</h3>
 * The &lt;action ... /&gt; configuration for this action is very straightforward.  The action just takes one
 * property value, which is the name of the JBossWS endpoint it's exposing (invoking).
 *
 * <pre>
 * &lt;action name="ShippingProcessor" class="org.jboss.soa.esb.actions.soap.SOAPProcessor"&gt;
 *     &lt;property name="<b>jbossws-endpoint</b>" value="<b>ABI_Shipping</b>"/&gt;
 * &lt;/action&gt;
 * </pre>
 * 
 * <h3>Quickstarts</h3>
 * A number of quickstarts that demonstrate how to use this action are available in the JBossESB
 * distribution (samples/quickstarts).  See the "webservice_jbossws_adapter_01" and "webservice_bpel"
 * quickstarts.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
@Publish(WebserviceContractPublisher.class)
public class SOAPProcessor extends AbstractActionPipelineProcessor {

    public static final String JBOSSWS_ENDPOINT = "jbossws-endpoint";

    private static ThreadLocal<Message> messageTL = new ThreadLocal<Message>();
    private String jbossws_endpoint;
    private MessagePayloadProxy payloadProxy;

    /**
     * Public constructor.
     * @param config Configuration.
     * @throws ConfigurationException "jbossws-endpoint" not specified.
     */
    public SOAPProcessor(ConfigTree config) throws ConfigurationException {
        jbossws_endpoint = config.getRequiredAttribute(JBOSSWS_ENDPOINT);
        payloadProxy = new MessagePayloadProxy(config,
                                               new String[] {BytesBody.BYTES_LOCATION, ActionUtils.POST_ACTION_DATA},
                                               new String[] {ActionUtils.POST_ACTION_DATA});
    }

    /**
     * Process the SOAP message.
     * <p/>
     * Invokes the JBossWS endpoint and writes the SOAP response back into the message payload.
     * @param message The ESB Aware (normalized) SOAP request message.
     * @return The SOAP response message.
     * @throws ActionProcessingException
     */
    public Message process(Message message) throws ActionProcessingException {
        Endpoint endpoint = getServiceEndpoint(jbossws_endpoint);
        byte[] soapMessage;
        String response = null;

        if(endpoint == null) {
            throw new ActionProcessingException("Unknown Service Endpoint '" + jbossws_endpoint + "'.");
        }

        soapMessage = getSOAPMessagePayload(message);
        try {
            messageTL.set(message);

            RequestHandler requestHandler = endpoint.getRequestHandler();
            InvocationContext invocationContext = new InvocationContext();
            ByteArrayOutputStream os = new ByteArrayOutputStream();

            requestHandler.handleRequest(endpoint, new ByteArrayInputStream(soapMessage), os, invocationContext);

            Object contentType = message.getProperties().getProperty("Content-Type");
            if(!(contentType instanceof ResponseHeader)) {
                message.getProperties().setProperty("Content-Type", new ResponseHeader("Content-Type", "text/xml"));
            }

            response = new String(os.toByteArray()).trim();
        } catch (Exception ex) {
            throw new ActionProcessingException("Cannot process SOAP request", ex);
        } finally {
            // Get the message instance set on the Threadlocal before removing it.  The Webservice endpoint
            // may have reset it with a new Message instance.
            message = messageTL.get();
            messageTL.remove();
            try {
                payloadProxy.setPayload(message, response);
            } catch (MessageDeliverException e) {
                throw new ActionProcessingException(e);
            }
        }

        return message;
    }

    private byte[] getSOAPMessagePayload(Message message) throws ActionProcessingException {
        byte[] soapMessage;
        Object messagePayload;

        try {
            messagePayload = payloadProxy.getPayload(message);
        } catch (MessageDeliverException e) {
            throw new ActionProcessingException(e);
        }

        if(messagePayload instanceof byte[]) {
            soapMessage = (byte[])messagePayload;
        } else if(messagePayload instanceof String) {
            try {
                soapMessage = ((String)messagePayload).getBytes("UTF-8");
            } catch (UnsupportedEncodingException e) {
                throw new ActionProcessingException("Unable to decode SOAP message payload.", e);
            }
        } else {
            throw new ActionProcessingException("Unable to decode SOAP message payload.  Must be either a byte[] or java.lang.String.");
        }
        return soapMessage;
    }

    /**
     * Set the {@link org.jboss.soa.esb.message.Message} instance for this invocation context.
     * <p/>
     * This allows message aware Webservice endpoints modify the {@link org.jboss.soa.esb.message.Message}
     * instance for the current Action Processing Pipeline.
     *
     * @param message The new message instance.
     */
    public static void setMessage(final Message message) {
        messageTL.set(message);
    }

    /**
     * Get the {@link org.jboss.soa.esb.message.Message} instance for this invocation context.
     *  
     * @return The message instance.
     */
    public static Message getMessage() {
        return messageTL.get();
    }

    /**
     * Get the Service Endpoint based on the endpoint name.
     * @param endpointName Service Endpoint name.
     * @return The service endpoint, or null if the endpoint is not found.
     */
    protected static Endpoint getServiceEndpoint(String endpointName) {
        SPIProvider spiProv = SPIProviderResolver.getInstance().getProvider();
        EndpointRegistryFactory factory =  spiProv.getSPI(EndpointRegistryFactory.class);
        EndpointRegistry registry = factory.getEndpointRegistry();
        Set<ObjectName> objectNames = registry.getEndpoints();

        for (ObjectName objectName : objectNames) {
            String endpoint = objectName.getKeyProperty(Endpoint.SEPID_PROPERTY_ENDPOINT);

            if (endpoint != null && endpoint.equals(endpointName)) {
                return registry.getEndpoint(objectName);
            }
        }

        return null;
    }
}