/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.admin.console.exchange;

import static org.jboss.seam.ScopeType.SESSION;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.ejb.Remove;
import javax.ejb.Stateful;
import javax.faces.model.SelectItem;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import static javax.persistence.PersistenceContextType.EXTENDED;

import org.apache.log4j.Logger;
import org.jboss.seam.annotations.Create;
import org.jboss.seam.annotations.Destroy;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Out;
import org.jboss.seam.annotations.Scope;
import org.jboss.seam.contexts.Contexts;
import org.jboss.soa.esb.admin.console.SeamViewIdStack;
import org.jboss.soa.esb.admin.console.contract.MessageContract;
import org.jboss.soa.esb.admin.console.contract.MessageContractDTO;
import org.jboss.soa.esb.admin.console.exchange.participant.Participant;

/**
 * Message Exchange Select Listener.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
@Stateful
@Name("messageExchangeSelect")
@Scope(SESSION)
public class SelectMessageExchangeAction implements Serializable, SelectMessageExchange {

	private static final long serialVersionUID = 1L;
	static Logger logger = Logger.getLogger(SelectMessageExchangeAction.class);

	@PersistenceContext(type=EXTENDED) 
	private EntityManager entityManager;
	
	@In(required=false) @Out(scope=SESSION) 
	private List<Participant> participants;
	
	private static List<MessageContractDTO> emptyContractList = new ArrayList<MessageContractDTO>();

	@In(required=false) @Out(scope=SESSION) MessageExchange messageExchange;
	
	@Create
	public void initialise() {
		messageExchange = new MessageExchange();
		participants = entityManager.createQuery("from Participant order by name").getResultList();
	}
	
	public String showFrom() {
		resetExchange();
		return "select-message-exchange-from";
	}
	
	public String showFinish() {
		return "select-message-exchange-finish";
	}
	
	public String finish() {
		setExchange();
		return SeamViewIdStack.getInstance().popViewId();
	}

	/* (non-Javadoc)
	 * @see org.jboss.soa.esb.admin.console.exchange.SelectMessageExchange#selectFrom(String)
	 */
	public List<MessageContractDTO> selectFrom(String selectionId) {
		Participant from = null;
		
		messageExchange.setFrom(null);
		messageExchange.setFromMessageType(null);
		from = findParticipant(selectionId, participants);
		if(from == null) {
			logger.error("Unexpected error.  Failed to locate 'from' exchange participant '" + selectionId + "'.  This may be OK.  Returning empty message contract list.");
			return emptyContractList;
		}
		messageExchange.setFrom(from);
		
		// Select the message contracts *produced* by the selected "from" Participant...
		List<MessageContract> contracts = entityManager.createQuery("from MessageContract where participant = :participant and usage = 'PRODUCES'")
										.setParameter("participant", from.getName())
										.getResultList();
		
		return MessageContract.toDTOList(contracts);
	}

	/**
	 * Select the "from" {@link MessageContract} action listener method.
	 */
	public void selectFromContract(String selectionId) {
		messageExchange.setFromMessageType(findContract(selectionId));
	}	

	/**
	 * Select the "to" {@link Participant} action listener method.
	 */
	public List<MessageContractDTO> selectTo(String selectionId) {
		Participant to = null;
		
		messageExchange.setTo(null);
		messageExchange.setToMessageType(null);
		to = findParticipant(selectionId, participants);
		if(to == null) {
			logger.warn("Unexpected error.  Failed to locate 'to' exchange participant '" + selectionId + "'.  This may be OK.  Returning empty message contract list.");
			return emptyContractList;
		}
		messageExchange.setTo(to);
		
		// Select the message contracts *produced* by the selected "from" Participant...
		List<MessageContract> contracts = entityManager.createQuery("from MessageContract where participant = :participant and usage = 'CONSUMES'")
										.setParameter("participant", to.getName())
										.getResultList();
		
		return MessageContract.toDTOList(contracts);
	}

	/**
	 * Select the "to" {@link MessageContract} action listener method.
	 */
	public void selectToContract(String selectionId) {
		messageExchange.setToMessageType(findContract(selectionId));
	}

	public String home() {
		resetExchange();
		
		return "home";
	}
	
	@Remove
	@Destroy
	public void destroy() {		
	}

	void setExchange() {
		Contexts.getSessionContext().set(MessageExchange.class.getName(), "set");
		Contexts.getSessionContext().set(MessageExchange.class.getName() + "#instance", messageExchange);
	}

	void resetExchange() {
		Contexts.getSessionContext().set(MessageExchange.class.getName(), "notset");
		initialise();
	}
	
	/**
	 * Is the {@link MessageExchange} selected on the current session.
	 * @return True if the {@link MessageExchange} is selected on the current session, otherwise false.
	 */
	public static boolean isMessageExchangeSelected() {
		String set = (String)Contexts.getSessionContext().get(MessageExchange.class.getName());
		
		return (set != null && set.equals("set"));
	}
	
	/**
	 * Get the current MessageExchange instance.
	 * @return The current MessageExchange instance.
	 */
	public static MessageExchange getMessageExchange() {
		if(isMessageExchangeSelected()) {
			return (MessageExchange) Contexts.getSessionContext().get(MessageExchange.class.getName() + "#instance");
		}
		
		throw new IllegalStateException("Call to 'getMessageExchange()' before an exchange has been selected.  The calling code needs to protect itself against this condition!");
	}

	public List<SelectItem> getParticipantListItems() {
		List<SelectItem> items = new ArrayList<SelectItem>();
		
		items.add(new SelectItem("All"));
		for(int i = 0; i < participants.size(); i++) {
			Participant participant = participants.get(i);
			items.add(new SelectItem(participant.getId(), participant.getName()));
		}
		
		return items;
	}

	public List<Participant> getParticipants() {
		return participants;
	}

	public void setParticipants(List<Participant> fromMessageParticipants) {
		this.participants = fromMessageParticipants;
	}

	Participant findParticipant(String id, List<Participant> participants) {
		for(Participant participant : participants) {
			if(participant.getId().equals(id)) {
				return participant;
			}
		}
		return null;
	}

	MessageContract findContract(String id) {
		MessageContract contracts = (MessageContract) entityManager.createQuery("from MessageContract where id = :id")
									.setParameter("id", id)
									.getSingleResult();

		return contracts;
	}
}
