/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.admin.console.transform;

import static org.jboss.seam.ScopeType.SESSION;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;

import javax.persistence.EntityManager;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.log4j.Logger;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Out;
import org.jboss.seam.annotations.Scope;
import org.jboss.soa.esb.admin.console.SeamViewIdStack;
import org.jboss.soa.esb.admin.console.exchange.MessageExchange;
import org.jboss.soa.esb.admin.console.exchange.SelectMessageExchangeAction;
import org.milyn.Smooks;
import org.milyn.SmooksException;
import org.milyn.SmooksUtil;
import org.milyn.profile.DefaultProfileSet;
import org.milyn.profile.ProfileStore;
import org.xml.sax.SAXException;

/**
 * Test Transformation action listener.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
@Name("testTransformation")
@Scope(SESSION)
public class TestTransformation implements Serializable {

	private static final long serialVersionUID = 1L;

	private static Logger logger = Logger.getLogger(TestTransformation.class);

	@In(create=true)
	private EntityManager entityManager;

	@In(required=false) @Out(scope=SESSION)
	private TestTransformationView testView;
	
	private byte[] testData;

	public String show() {
		if(testView == null) {
			// How do we get Seam to autocreate the instance?? Tried the create=true annotation but no joy!
			testView = new TestTransformationView();
		}
		
		if(!SelectMessageExchangeAction.isMessageExchangeSelected()) {
			SeamViewIdStack.getInstance().pushViewId("test-transformation");
			return "select-message-exchange";
		}
		
		return "test-transformation";
	}
	
	public void applyResources() throws UnsupportedEncodingException {
        Smooks smooks = new Smooks();
        ByteArrayInputStream config = new ByteArrayInputStream(SmooksResourceList.getList(entityManager).getBytes("UTF-8"));
        ByteArrayInputStream input = new ByteArrayInputStream(testView.getInput().getBytes("UTF-8"));
        MessageExchange exchange = SelectMessageExchangeAction.getMessageExchange();
        
        try {
        	smooks.addConfigurations("ConsoleResources", config);
		} catch (Throwable t) {
			logger.error("Error loading transformation resources.", t);
			testView.setOutput("Error Loading Transformation Resources...\n\n" + t.getClass().getName() + ": " + t.getMessage() + "\n\nCheck Java console.");
			return;
		}
		
        try {
        	String[] profiles = exchange.getUseragentProfiles();
    		SmooksUtil.registerProfileSet(DefaultProfileSet.create( exchange.getExchangeUseragentString(), profiles), smooks);
        	
        	if(profiles == null || profiles.length == 0) {
	        	ProfileStore profileStore = smooks.getApplicationContext().getProfileStore();
	        	profileStore.getProfileSet(exchange.getExchangeUseragentString());
        	} else {
	        	SmooksUtil.registerProfileSet(DefaultProfileSet.create(exchange.getExchangeUseragentString(), profiles ), smooks);
        	}
		} catch (Throwable t) {
			logger.error("Error registering Smooks message profiles.", t);
			testView.setOutput("Error registering Smooks message profiles...\n\n" + t.getClass().getName() + ": " + t.getMessage() + "\n\nCheck Java console.");
			return;
		}
		
        try {
        	String output = SmooksUtil.filterAndSerialize(smooks.createExecutionContext(),input, smooks );
        	testView.setOutput(format(output));
		} catch (Throwable t) {
			logger.error("Error in applying Transformation/Analysis resources to message.", t);
			if(t instanceof SmooksException && t.getCause() instanceof SAXException) {
				testView.setOutput("Transformation Exception...\n\n" + t.getClass().getName() + ": " + t.getMessage() + "\n\nCheck Java console.\n\nNOTE: This particular error was caused by a " + t.getCause().getClass().getName() + " exception which can result from attempting to parse the incomming message with the wrong SAX Parser.  If the message is not an XML message, you must specify an appropriate parser.");
			} else {
				testView.setOutput("Transformation Exception...\n\n" + t.getClass().getName() + ": " + t.getMessage() + "\n\nCheck Java console.");
			}
		}
	}

	private static TransformerFactory factory = TransformerFactory.newInstance();

	/**
	 * Attempt to format the output.
	 * @param output Output to be formatted.
	 * @return Output.
	 */
	private String format(String output) {
		StreamSource source = new StreamSource(new ByteArrayInputStream(output.getBytes()));
		ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		
		try {
			Transformer transformer;
			
			// There's a bug in Java 5 re this code (formatting).
			// See http://forum.java.sun.com/thread.jspa?threadID=562510&start=0 and it explains the
			// whys of the following code.
			transformer = factory.newTransformer();
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");			
			transformer.setOutputProperty("{http://xml.apache.org/xalan}indent-amount", "4");
			transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
			transformer.transform(source, new StreamResult(outStream));
			
			return outStream.toString();
		} catch (Exception e) {
			return output;
		}
	}
	
	public void loadTestData() {
		if(testData != null) {
			testView.setInput(new String(testData));
		}
	}
	
	public byte[] getTestData() {
		return testData;
	}

	public void setTestData(byte[] testData) {
		this.testData = testData;
	}
}
