/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.server.jcaTx;

import javax.jms.Connection;
import javax.jms.ConnectionFactory;
import javax.jms.MessageProducer;
import javax.jms.Queue;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.jms.TextMessage;
import javax.management.ObjectName;
import javax.naming.InitialContext;

import junit.framework.Test;

import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.server.RedeliveryMBean;
import org.jboss.test.JBossTestCase;

/**
 * Unit test for JCA TX tests.
 *
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */

public class JcaTXUnitTest extends JBossTestCase
{
    public JcaTXUnitTest(final String name)
    {
        super(name);
    }

    public void sendATextMessage(final String msg, final String destination)
        throws Exception
    {
        final InitialContext initCtx = getInitialContext();
        final ConnectionFactory connectionFactory = (ConnectionFactory) initCtx.lookup("ConnectionFactory") ;
        final Connection connection = connectionFactory.createConnection() ;
        final Queue queue = (Queue)initCtx.lookup(destination) ;
        final Session session = connection.createSession(false, QueueSession.AUTO_ACKNOWLEDGE) ;
        connection.start();
        final MessageProducer messageProducer = session.createProducer(queue) ;
        final TextMessage message = session.createTextMessage(msg) ;
        messageProducer.send(message) ;
        messageProducer.close() ;
        connection.close() ;
    }

    public void sendAnESBMessage(final String msg, final String category, final String service)
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        message.getBody().add(msg) ;
        final ServiceInvoker serviceInvoker = new ServiceInvoker(category, service) ;
        serviceInvoker.deliverAsync(message) ;
    }

    public void testJcaJmsGatewayRedelivery()
        throws Exception
    {
        final String message = "Hello TX Gateway World" ;
        sendATextMessage(message, "queue/esb_jca_tx_channel") ;
        
        final String[] messages = (String[])getServer().invoke(new ObjectName(RedeliveryMBean.objectName), "waitForMessages",
            new Integer[] {Integer.valueOf(4)}, new String[] { Integer.TYPE.getName() });
        final int numMessages = (messages == null ? 0 : messages.length) ;
        
        assertEquals("Message count", 4, numMessages) ;
        assertEquals("first message", message, messages[0]) ;
        assertEquals("second message",
            "Exception for message " + message + ":" + "An error to force redelivery",
            messages[1]) ;
        assertEquals("third message", message, messages[2]) ;
        assertEquals("fourth message",
            "Success for message " + message,
            messages[3]) ;
    }

    public void testJcaJmsESBAwareRedelivery()
        throws Exception
    {
        final String message = "Hello TX ESB World" ;
        sendAnESBMessage(message, "HelloWorld_ActionESB", "SimpleListener") ;
        
        final String[] messages = (String[])getServer().invoke(new ObjectName(RedeliveryMBean.objectName), "waitForMessages",
            new Integer[] {Integer.valueOf(4)}, new String[] { Integer.TYPE.getName() });
        final int numMessages = (messages == null ? 0 : messages.length) ;
        
        assertEquals("Message count", 4, numMessages) ;
        assertEquals("first message", message, messages[0]) ;
        assertEquals("second message",
            "Exception for message " + message + ":" + "An error to force redelivery",
            messages[1]) ;
        assertEquals("third message", message, messages[2]) ;
        assertEquals("fourth message",
            "Success for message " + message,
            messages[3]) ;
    }

    public static Test suite() throws Exception
    {
        return getDeploySetup(JcaTXUnitTest.class, "jca-tx-test.esb");
    }
}
