/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.integration.server;

import javax.jms.JMSException;
import javax.naming.NamingException;
import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmContext;
import org.jbpm.bpel.BpelException;
import org.jbpm.bpel.app.AppDescriptor;
import org.jbpm.bpel.graph.def.BpelProcessDefinition;
import org.jbpm.bpel.integration.jms.IntegrationControl;
import org.jbpm.bpel.integration.jms.JmsIntegrationServiceFactory;

/**
 * @author Alejandro Guizar
 * @version $Revision: 1.5 $ $Date: 2007/07/20 22:29:54 $
 */
public class IntegrationConfigurator implements ServletContextListener {

  /** Initialization parameter for the jBPM configuration resource. */
  public static final String JBPM_CONFIG_RESOURCE_PARAM = "JbpmCfgResource";
  
  private static final Log log = LogFactory.getLog(IntegrationConfigurator.class);

  public void contextInitialized(ServletContextEvent event) {
    ServletContext servletContext = event.getServletContext();

    String configurationName = servletContext.getInitParameter(JBPM_CONFIG_RESOURCE_PARAM);
    JbpmConfiguration jbpmConfiguration;
    if (configurationName != null)
      jbpmConfiguration = JbpmConfiguration.getInstance(configurationName);
    else
      jbpmConfiguration = JbpmConfiguration.getInstance();

    JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
    try {
      // read the app descriptor from a classpath resource
      AppDescriptor appDescriptor = AppDescriptor.readAppDescriptor(jbpmContext);
      /*
       * obtain the integration control associated with the process definition referenced in the
       * descriptor
       */
      BpelProcessDefinition processDefinition = appDescriptor.findProcessDefinition(jbpmContext);
      JmsIntegrationServiceFactory integrationServiceFactory = JmsIntegrationServiceFactory.getInstance(jbpmConfiguration);
      IntegrationControl integrationControl = integrationServiceFactory.getIntegrationControl(processDefinition);

      // make app descriptor available to message activities
      integrationControl.setAppDescriptor(appDescriptor);
      // start receiving requests
      integrationControl.enableInboundMessageActivities(jbpmContext);

      // make integration control available to jax-rpc handlers
      servletContext.setAttribute(SoapHandler.INTEGRATION_CONTROL_ATTR, integrationControl);

      log.info("Message reception enabled for process: " + appDescriptor.getName());
    }
    catch (RuntimeException e) {
      jbpmContext.setRollbackOnly();
      throw e;
    }
    catch (NamingException e) {
      jbpmContext.setRollbackOnly();
      throw new BpelException("could not start bpel application", e);
    }
    catch (JMSException e) {
      jbpmContext.setRollbackOnly();
      throw new BpelException("could not start bpel application", e);
    }
    finally {
      jbpmContext.close();
    }
  }

  public void contextDestroyed(ServletContextEvent event) {
    IntegrationControl integrationControl = (IntegrationControl) event.getServletContext()
        .getAttribute(SoapHandler.INTEGRATION_CONTROL_ATTR);
    // context initialization might have failed to retrieve the integration control
    if (integrationControl != null) {
      try {
        integrationControl.disableInboundMessageActivities();
        log.info("Message reception disabled for process: "
            + integrationControl.getAppDescriptor().getName());
      }
      catch (JMSException e) {
        log.error("could not stop bpel application", e);
      }
    }
  }
}
