/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.graph.exe.state;

import java.util.Date;

import junit.framework.TestCase;

import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmContext;
import org.jbpm.bpel.graph.basic.Empty;
import org.jbpm.bpel.graph.def.Activity;
import org.jbpm.bpel.graph.def.BpelProcessDefinition;
import org.jbpm.bpel.graph.exe.FaultInstance;
import org.jbpm.bpel.graph.exe.ScopeInstance;
import org.jbpm.bpel.graph.exe.ScopeState;
import org.jbpm.bpel.graph.scope.Scope;
import org.jbpm.bpel.graph.struct.Flow;
import org.jbpm.bpel.graph.struct.Sequence;
import org.jbpm.bpel.xml.BpelConstants;
import org.jbpm.graph.exe.ExecutionContext;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.graph.exe.Token;

/**
 * @author Juan Cantu
 * @version $Revision: 1.8 $ $Date: 2007/08/08 11:09:33 $
 */
public abstract class AbstractStateTestCase extends TestCase {

  Token root;
  ScopeInstance scopeInstance;
  TestScopeInstance parent;
  TestScopeInstance activeChild;
  TestScopeInstance handlingChild;
  TestScopeInstance completedChild;
  BpelProcessDefinition pd;
  Scope scope;
  Scope parentScope;
  LogActivity scopeCompletionLog;
  LogActivity handlerLog;

  private JbpmContext jbpmContext;

  protected void setUp() {
    JbpmConfiguration jbpmConfiguration = JbpmConfiguration.getInstance("org/jbpm/bpel/graph/exe/test.jbpm.cfg.xml");
    jbpmContext = jbpmConfiguration.createJbpmContext();

    pd = new BpelProcessDefinition("process", BpelConstants.NS_EXAMPLES);

    Sequence rootSequence = new Sequence();
    pd.getGlobalScope().setActivity(rootSequence);

    // parent
    parentScope = new Scope();
    parentScope.installFaultExceptionHandler();
    parentScope.setName("parent");
    rootSequence.addNode(parentScope);
    scopeCompletionLog = new LogActivity();
    rootSequence.addNode(scopeCompletionLog);

    // parent instance
    ProcessInstance pi = new ProcessInstance(pd);
    root = pi.getRootToken();
    parent = createScopeInstance(parentScope, root);
    root.setNode(parentScope);

    // scope definition
    scope = new Scope();
    scope.installFaultExceptionHandler();
    scope.setName("scope");
    parentScope.addNode(scope);

    // scope instance
    Token scopeToken = new Token(root, "scope");
    scopeInstance = scope.createInstance(scopeToken);
    scopeInstance.setState(getState());
    scopeToken.setNode(scope);

    // children definition - instances
    Flow flow = new Flow();
    scope.setActivity(flow);

    Token activityToken = scopeInstance.getPrimaryToken();
    activityToken.setNode(flow);

    Scope activeScope = new Scope("active");
    activeScope.setActivity(new Empty());
    flow.addNode(activeScope);
    Token forkToken = new Token(activityToken, "1");
    forkToken.setNode(activeScope);
    Token aChild = new Token(forkToken, "active");
    activeChild = createScopeInstance(activeScope, aChild);
    activeChild.setState(ActiveState.PERFORMING_PRIMARY_ACTIVITY);

    Scope handlingScope = new Scope("handling");
    handlingScope.setActivity(new Empty());
    flow.addNode(handlingScope);
    forkToken = new Token(activityToken, "2");
    forkToken.setNode(handlingScope);
    aChild = new Token(forkToken, "handling");
    handlingChild = createScopeInstance(handlingScope, aChild);
    handlingChild.setState(FaultingState.FAULTING_WITH_HANDLER);

    Scope completedScope = new Scope("completed");
    completedScope.setActivity(new Empty());
    flow.addNode(completedScope);
    forkToken = new Token(activityToken, "3");
    forkToken.setNode(completedScope);
    aChild = new Token(forkToken, "completed");
    completedChild = createScopeInstance(completedScope, aChild);
    // completed token MUST be null
    completedChild.getToken().end();
    completedChild.setState(EndedState.COMPLETED);

    handlerLog = new LogActivity();
  }

  protected void tearDown() throws Exception {
    jbpmContext.close();
  }

  public abstract ScopeState getState();

  public void testFaulted() {
    try {
      scopeInstance.faulted(null);
      fail("faulted can't be invoked at this state");
    }
    catch (IllegalStateException e) {
      // expected exception
    }
  }

  public void testTerminate() {
    try {
      scopeInstance.terminate();
      fail("terminate can't be invoked at this state");
    }
    catch (IllegalStateException e) {
      // expected exception
    }
  }

  public void testCompleted() {
    try {
      scopeInstance.completed();
      fail("completed can't be invoked at this state");
    }
    catch (IllegalStateException e) {
      // expected exception
    }
  }

  public void testCompensate() {
    try {
      scopeInstance.compensate(null);
      fail("compensate can't be invoked at this state");
    }
    catch (IllegalStateException e) {
      // expected exception
    }

  }

  public void testChildrenTerminated() {
    try {
      scopeInstance.getState().childrenTerminated(scopeInstance);
      fail("children terminated can't be invoked at this state");
    }
    catch (IllegalStateException e) {
      // expected exception
    }
  }

  // compensation won't work since it depends on persistence
  void assertChildrenCompensated() {
    assertNull(activeChild.compensated);
    assertNull(handlingChild.compensated);
    assertNotNull(completedChild.compensated);
  }

  void assertChildrenTerminated() {
    assertTrue(activeChild.terminated);
    assertTrue(handlingChild.terminated);
    assertFalse(completedChild.terminated);
  }

  static TestScopeInstance createScopeInstance(Scope scope, Token token) {
    TestScopeInstance scopeInstance = new TestScopeInstance(scope, token);
    token.getProcessInstance().getContextInstance().createVariable(Scope.VARIABLE_NAME,
        scopeInstance, token);
    return scopeInstance;
  }

  static class TestScopeInstance extends ScopeInstance {

    boolean childCompensated = false;
    boolean childTerminated = false;
    boolean childFaulted = false;

    boolean terminated = false;
    Date compensated;

    private static final long serialVersionUID = 1L;

    TestScopeInstance(Scope scope, Token token) {
      super(scope, token);
    }

    public void scopeCompensated(ScopeInstance child) {
      childCompensated = true;
    }

    public void scopeTerminated(ScopeInstance child) {
      childTerminated = true;
    }

    public void faulted(FaultInstance faultInstance) {
      childFaulted = true;
    }

    public void terminate() {
      terminated = true;
    }

    public void compensate() {
      compensated = new Date();
    }
  }

  static class LogActivity extends Activity {

    boolean executed = false;

    private static final long serialVersionUID = 1L;

    public void execute(ExecutionContext context) {
      executed = true;
    }
  }

}
