/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.tools;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * A collection of utility methods on {@linkplain File files}.
 * @author Alejandro Guizar
 * @version $Revision: 1.5 $ $Date: 2007/09/05 08:25:10 $
 */
public class FileUtil {

  private static ThreadLocal transferBufferLocal = new ThreadLocal() {

    protected Object initialValue() {
      return new byte[512];
    }
  };

  // suppress default constructor, ensuring non-instantiability
  private FileUtil() {
  }

  /**
   * The default temp file path, provided by the <code>java.io.tmpdir</code> system property.
   */
  public static final File TEMPORARY_DIRECTORY = new File(System.getProperty("java.io.tmpdir"));

  /**
   * The user's current working directory, provided by the <code>user.dir</code> system property.
   */
  public static final File CURRENT_DIRECTORY = new File(System.getProperty("user.dir"));

  /**
   * Recursively deletes the contents of the given directory.
   * @param dir the directory to delete; if <code>dir</code> does not denote a directory, then
   * this method performs no deletion
   * @return <code>true</code> if and only if the directory is completely deleted;
   * <code>false</code> otherwise
   */
  public static boolean deleteDirectory(File dir) {
    File[] files = dir.listFiles();

    if (files == null) {
      // dir is NOT a directory, or I/O error
      return false;
    }

    // delete files in the directory
    for (int i = 0; i < files.length; i++) {
      File file = files[i];

      // TODO break on the first failed deletion?
      if (file.isDirectory())
        deleteDirectory(file);
      else
        file.delete();
    }

    // lastly, delete the directory itself
    return dir.delete();
  }

  /**
   * Adds the contents of the given file or directory to the zip stream. If the argument is a
   * directory, then the files it contains are recursively zipped, preserving the relative path
   * hierarchy. The given prefix is prepended to the name of the file or directory to construct the
   * name of the zip entry.
   * @param path the file or directory to zip
   * @param zipOut the stream where data is written
   * @param prefix the zip entry prefix
   * @throws IOException if an I/O error occurs
   */
  public static void zip(File path, ZipOutputStream zipOut, String prefix) throws IOException {
    if (path.isDirectory())
      zipDirectory(path, zipOut, prefix);
    else
      zipFile(path, zipOut, prefix);
  }

  /**
   * Recursively adds the contents of the given directory to the zip stream. The given prefix is
   * prepended to the directory name.
   * @param dir the directory to zip
   * @param zipOut the stream where data is written
   * @param prefix the zip entry prefix
   * @throws IOException if an I/O error occurs
   */
  private static void zipDirectory(File dir, ZipOutputStream zipOut, String prefix)
      throws IOException {
    File[] files = dir.listFiles();
    if (files == null)
      throw new IOException("file listing failed: " + dir);

    // add files in the directory
    for (int i = 0; i < files.length; i++) {
      File file = files[i];

      if (file.isDirectory())
        zipDirectory(file, zipOut, prefix + WebModuleBuilder.SEPARATOR + file.getName());
      else
        zipFile(file, zipOut, prefix);
    }
  }

  /**
   * Adds the content of the given file to the zip stream. The given prefix is prepended to the file
   * name.
   * @param dir the file to zip
   * @param zipOut the stream where data is written
   * @param prefix the zip entry prefix
   * @throws IOException if an I/O error occurs
   */
  private static void zipFile(File file, ZipOutputStream zipOut, String prefix) throws IOException {
    InputStream fileSource = new FileInputStream(file);
    try {
      zipOut.putNextEntry(new ZipEntry(prefix + WebModuleBuilder.SEPARATOR + file.getName()));

      // transfer bytes from the file to the archive using our thread local byte buffer
      byte[] transferBuffer = (byte[]) transferBufferLocal.get();
      for (int length; (length = fileSource.read(transferBuffer)) > 0;)
        zipOut.write(transferBuffer, 0, length);

      zipOut.closeEntry();
    }
    finally {
      fileSource.close();
    }
  }
}
