package org.jbpm.gd.jpdl.ui.properties;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.Action;
import org.jbpm.gd.jpdl.model.ConfigInfoElement;
import org.jbpm.gd.jpdl.ui.dialog.ChooseDelegationClassDialog;

public class ActionConfigurationComposite implements SelectionListener, FocusListener {
	
	public static ActionConfigurationComposite create(TabbedPropertySheetWidgetFactory widgetFactory, Composite parent) {
		ActionConfigurationComposite result = new ActionConfigurationComposite();
		result.widgetFactory = widgetFactory;
		result.parent = parent;
		result.create();
		return result;
	}
	
	private TabbedPropertySheetWidgetFactory widgetFactory;
	private Composite parent;
	
	private Group actionInfoGroup;
	private Group handlerInfoGroup;
	private Group expressionInfoGroup;
	
	private Label nameLabel;
	private Text nameText;
	private Button refNameButton;
	private Button acceptPropagatedEventsButton;
	private Button asyncButton;
	private Label actionTypeLabel;
	private CCombo actionTypeCombo;
	
	private Label expressionLabel;
	private Text expressionText;
	private DelegationConfigurationComposite delegationConfigurationComposite;
	
	private Action action;
	private String selectedActionType = "";
	
	private ActionConfigurationComposite() {}
	
	public void setAction(Action action) {
		if (this.action == action) return;
		unhookListeners();
		this.action = action;
		delegationConfigurationComposite.setDelegation(action);
		if (action == null) {
			clearControls();
		} else {
			updateControls();
			hookListeners();
		}
	}
	
	public Action getAction() {
		return action;
	}
	
	private void hookListeners() {
		nameText.addSelectionListener(this);
		nameText.addFocusListener(this);
		refNameButton.addSelectionListener(this);
		acceptPropagatedEventsButton.addSelectionListener(this);
		asyncButton.addSelectionListener(this);
		actionTypeCombo.addSelectionListener(this);
		expressionText.addSelectionListener(this);
		expressionText.addFocusListener(this);
	}
	
	private void unhookListeners() {
		nameText.removeSelectionListener(this);
		nameText.removeFocusListener(this);
		refNameButton.removeSelectionListener(this);
		acceptPropagatedEventsButton.removeSelectionListener(this);
		asyncButton.removeSelectionListener(this);
		actionTypeCombo.removeSelectionListener(this);
		expressionText.removeSelectionListener(this);
		expressionText.removeFocusListener(this);
	}
	
	private void clearControls() {
		nameText.setText("");
		refNameButton.setSelection(false);
		handleRefNameButtonSelected();
		acceptPropagatedEventsButton.setSelection(false);
		asyncButton.setSelection(false);
		actionTypeCombo.setText("");
		handleActionTypeComboSelected();
		expressionText.setText("");
	}
	
	private void updateControls() {
		String name = action.getName();
		String refName = action.getRefName();
		refNameButton.setSelection(refName != null);
		if (refName != null) {
			name = refName;
		}
		nameText.setText(name == null ? "" : name);
		acceptPropagatedEventsButton.setSelection("true".equals(action.getAcceptPropagatedEvents()));
		asyncButton.setSelection("true".equals(action.getAsync()));
		String expression = action.getExpression();
		expressionText.setText(expression == null ? "" : expression);
		if (action.getClassName() != null) {
			actionTypeCombo.setText("handler");
		} else if (expression != null) {
			actionTypeCombo.setText("expression");
		} else {
			actionTypeCombo.setText("");
		}
		updateControlVisibility();
	}
	
	private void create() {
		createMainInfoGroup();
		createHandlerInfoGroup();
		createExpressionInfoGroup();
		initializeLayouts();
	}
	
	private void createMainInfoGroup() {
		actionInfoGroup = widgetFactory.createGroup(parent, "Action Information");
		actionInfoGroup.setLayout(new FormLayout());
		nameLabel = widgetFactory.createLabel(actionInfoGroup, "Name");
		nameText = widgetFactory.createText(actionInfoGroup, "");
		refNameButton = widgetFactory.createButton(actionInfoGroup, "Reference", SWT.CHECK);
		refNameButton.setSelection(false);
		refNameButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleRefNameButtonSelected();
			}
		});
		acceptPropagatedEventsButton = widgetFactory.createButton(actionInfoGroup, "Accept Propagated Events", SWT.CHECK);
		asyncButton = widgetFactory.createButton(actionInfoGroup, "Asynchrounous", SWT.CHECK);
		actionTypeLabel = widgetFactory.createLabel(actionInfoGroup, "Action Type");
		actionTypeCombo = widgetFactory.createCCombo(actionInfoGroup, SWT.BORDER);
		actionTypeCombo.setItems(new String[] {"", "expression", "handler"});
		actionTypeCombo.setEditable(false);
		actionTypeCombo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleActionTypeComboSelected();
			}
		});
	}
	
	private void createHandlerInfoGroup() {
		handlerInfoGroup = widgetFactory.createGroup(parent, "Handler Information");
		handlerInfoGroup.setLayout(new FormLayout());
		handlerInfoGroup.setVisible(false);
		delegationConfigurationComposite = DelegationConfigurationComposite.create(widgetFactory, handlerInfoGroup);
		delegationConfigurationComposite.setChooseDelegationClassDialog(createChooseActionHandlerDialog());
	}
	
	private ChooseDelegationClassDialog createChooseActionHandlerDialog() {
		return new ChooseDelegationClassDialog(
				null, 
				"org.jbpm.graph.def.ActionHandler",
				"Choose Action Handler",
				"Choose an action handler from the list");
	}
	
	private void createExpressionInfoGroup() {
		expressionInfoGroup = widgetFactory.createGroup(parent, "Expression Information");
		expressionInfoGroup.setLayout(new FormLayout());
		expressionInfoGroup.setVisible(false);
		expressionLabel = widgetFactory.createLabel(expressionInfoGroup, "Expression");
		expressionText = widgetFactory.createText(expressionInfoGroup, "");
	}
	
	private void initializeLayouts() {
		nameLabel.setLayoutData(createNameLabelLayoutData());
		nameText.setLayoutData(createNameTextLayoutData());
		refNameButton.setLayoutData(createRefNameButtonLayoutData());
		acceptPropagatedEventsButton.setLayoutData(createAcceptPropagatedEventsButtonLayoutData());
		asyncButton.setLayoutData(createAsyncButtonLayoutData());
		actionTypeLabel.setLayoutData(createActionTypeLabelLayoutData());
		actionTypeCombo.setLayoutData(createActionTypeComboLayoutData());
		actionInfoGroup.setLayoutData(createActionInfoGroupLayoutData());
		handlerInfoGroup.setLayoutData(createDelegationInfoGroupLayoutData());
		expressionInfoGroup.setLayoutData(createDelegationInfoGroupLayoutData());
		expressionLabel.setLayoutData(createExpressionLabelLayoutData());
		expressionText.setLayoutData(createExpressionTextLayoutData());
	}
	
	private FormData createExpressionLabelLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(0, 5);
		result.left = new FormAttachment(0, 5);
		return result;
	}
	
	private FormData createExpressionTextLayoutData() {
		FormData result = new FormData();
		result.top = new FormAttachment(0, 5);
		result.left = new FormAttachment(expressionLabel, 5);
		result.right = new FormAttachment(100, -5);
		result.bottom = new FormAttachment(100, -5);
		return result;
	}
	
	private FormData createNameLabelLayoutData() {
		FormData result = new FormData();
		result.left = new FormAttachment(0, 5);
		result.bottom = new FormAttachment(actionTypeCombo, -5);
		return result;
	}
	
	private FormData createNameTextLayoutData() {
		FormData result = new FormData();
		result.left = new FormAttachment(actionTypeCombo);
		result.left.alignment = SWT.LEFT;
		result.top = new FormAttachment(0, 5);
		result.right = new FormAttachment(actionTypeCombo);
		result.right.alignment = SWT.RIGHT;
		result.bottom = new FormAttachment(actionTypeCombo, -5);
		return result;
	}
	
	private FormData createRefNameButtonLayoutData() {
		FormData result = new FormData();
		result.left = new FormAttachment(nameText, 5);
		result.bottom = new FormAttachment(actionTypeCombo, -5);
		return result;
	}
	
	private FormData createAcceptPropagatedEventsButtonLayoutData() {
		FormData result = new FormData();
		result.right = new FormAttachment(asyncButton, -5);
		result.bottom = new FormAttachment(100, -5);
		return result;
	}
	
	private FormData createAsyncButtonLayoutData() {
		FormData result = new FormData();
		result.bottom = new FormAttachment(100, -5);
		result.right = new FormAttachment(100, -5);
		return result;
	}
	
	private FormData createActionTypeLabelLayoutData() {
		FormData result = new FormData();
		result.left = new FormAttachment(0, 5);
		result.bottom = new FormAttachment(100, -5);
		return result;
	}
	
	private FormData createActionTypeComboLayoutData() {
		FormData result = new FormData();
		result.left = new FormAttachment(actionTypeLabel, 5);
		result.right = new FormAttachment(acceptPropagatedEventsButton, -5);
		result.bottom = new FormAttachment(100, -5);
		return result;
	}
	
	private FormData createActionInfoGroupLayoutData() {
		FormData result = new FormData();
		result.left = new FormAttachment(0, 5);
		result.top = new FormAttachment(0, 5);
		result.right = new FormAttachment(100, -5);
		return result;
	}
	
	private FormData createDelegationInfoGroupLayoutData() {
		FormData result = new FormData();
		result.left = new FormAttachment(0, 5);
		result.top = new FormAttachment(actionInfoGroup, 5);
		result.bottom = new FormAttachment(100, -5);
		result.right = new FormAttachment(100, -5);
		return result;
	}

	private String getNameText() {
		String text = nameText.getText();
		if (text != null && "".equals(text.trim())) {
			text = null;
		}
		return text;
	}
	
	private void updateActionName() {
		if (refNameButton.getSelection()) {
			action.setRefName(getNameText());
		} else {
			action.setName(getNameText());
		}
	}
	
	private void updateExpressionText() {
		action.setExpression(expressionText.getText());
	}
	
	private void handleAcceptPropagatedEventsButtonSelected() {
		action.setAcceptPropagatedEvents(getAcceptPropagatedEventsText());
	}
	
	private String getAcceptPropagatedEventsText() {
		return acceptPropagatedEventsButton.getSelection() ? "true" : "false";
	}
	
	private void handleAsyncButtonSelected() {
		action.setAsync(getAsyncText());
	}
	
	private String getAsyncText() {
		return asyncButton.getSelection() ? "true" : "false";
	}
	
	private void handleRefNameButtonSelected() {
		updateControlVisibility();
		if (action != null) {
			if (refNameButton.getSelection()) {
				removeAllActionInfo();
				action.setRefName(getNameText());
			} else {
				action.setRefName(null);
				restoreAllActionInfo();
			}
		}
	}

	private void updateControlVisibility() {
		boolean selected = refNameButton.getSelection();
		acceptPropagatedEventsButton.setVisible(!selected);
		asyncButton.setVisible(!selected);
		actionTypeLabel.setVisible(!selected);
		actionTypeCombo.setVisible(!selected);
		handlerInfoGroup.setVisible(!selected && "handler".equals(actionTypeCombo.getText()));
		expressionInfoGroup.setVisible(!selected && "expression".equals(actionTypeCombo.getText()));
	}
	
	private void restoreAllActionInfo() {
		if (refNameButton.getSelection()) {
			action.setRefName(getNameText());
		} else {
			action.setName(getNameText());
		}
		action.setAcceptPropagatedEvents(getAcceptPropagatedEventsText());
		action.setAsync(getAsyncText());
		updateExpressionAndHandlerInfo();
	}
	
	private void removeAllActionInfo() {
		action.setName(null);
		action.setRefName(null);
		action.setAcceptPropagatedEvents("true");
		action.setAsync("false");
		removeExpressionAndHandlerInfo();
	}

	private void handleActionTypeComboSelected() {
		String selection = actionTypeCombo.getText();
		if (selection.equals(selectedActionType)) return;
		selectedActionType = selection;
		updateControlVisibility();
		if (action != null) {
			removeExpressionAndHandlerInfo();
			updateExpressionAndHandlerInfo();
		}
	}

	private void updateExpressionAndHandlerInfo() {
		if ("expression".equals(selectedActionType)) {
			action.setExpression(expressionText.getText());
		} else if ("handler".equals(selectedActionType)) {
			delegationConfigurationComposite.updateDelegationInfo(action);
		}
	}
	
	private void removeExpressionAndHandlerInfo() {
		action.setExpression(null);
		action.setClassName(null);
		action.setConfigInfoString(null);
		action.setConfigType(null);
		ConfigInfoElement[] configElements = action.getConfigInfoElements();
		for (int i = 0; i < configElements.length; i++) {
			action.removeConfigInfoElement(configElements[i]);
		}
	}

	public void focusLost(FocusEvent e) {
		if (e.widget == nameText) {
			updateActionName();
		} else if (e.widget == expressionText) {
			updateExpressionText();
		}		
	}

	public void focusGained(FocusEvent e) {
	}
	
	public void widgetDefaultSelected(SelectionEvent e) {
		if (e.widget == nameText) {
			updateActionName();
		} else if (e.widget == expressionText) {
			updateExpressionText();
		}
	}

	public void widgetSelected(SelectionEvent e) {
		if (e.widget == refNameButton) {
			handleRefNameButtonSelected();
		} else if (e.widget == acceptPropagatedEventsButton) {
			handleAcceptPropagatedEventsButtonSelected();
		} else if (e.widget == asyncButton) {
			handleAsyncButtonSelected();
		} else if (e.widget == actionTypeCombo) {
			handleActionTypeComboSelected();
		}		
	}
	
}
