/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.remoting;

import org.jboss.remoting.transport.web.WebUtil;
import org.jboss.remoting.transport.http.HTTPMetadataConstants;

import java.util.Map;
import java.util.List;

/**
 * JBoss Remoting utility methods.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class JBossRemotingUtil {

    /**
     * Check if the server should expect a Java Object payload.
     * <p/>
     * Only relevant if the client is a JBoss Remoting client.
     *
     * @param metadata Request metadata.
     * @return True if the metadata indicates that the client/server is/should expect a Java Object
     * payload, otherwise false.
     */
    public static boolean receiveJavaObjectPayload(Map metadata) {
        return isJavaPayloadToFromJBRClient(metadata, (String) metadata.get(HTTPMetadataConstants.REMOTING_USER_AGENT));
    }

    /**
     * Check if the server should send a Java Object payload.
     * <p/>
     * Only relevant if the client is a JBoss Remoting client.
     * 
     * @param metadata Response metadata.
     * @param clientUserAgent Client User Agent.
     * @return True if the metadata indicates that the client/server is/should expect a Java Object
     * payload, otherwise false.
     */
    public static boolean sendJavaObjectPayload(Map metadata, String clientUserAgent) {
        return isJavaPayloadToFromJBRClient(metadata, clientUserAgent);
    }

    private static boolean isJavaPayloadToFromJBRClient(Map metadata, String clientUserAgent) {
        if (metadata != null) {
            if(clientUserAgent != null && clientUserAgent.startsWith("JBossRemoting")) {
                String value = getContentType(metadata);
                if (value != null) {
                    return WebUtil.isBinary(value);
                }
            }
        }

        return false;
    }

    /**
     * Get the content type from the supplied metadata.
     * @param metadata The request/response metadata.
     * @return The content type if set, otherwise null.
     */
    public static String getContentType(Map metadata) {
        if(metadata == null) {
            return null;
        }

        Object value = metadata.get("Content-Type");

        if (value == null) {
            value = metadata.get("content-type");
        }

        if (value != null) {
            if (value instanceof List) {
                List valueList = (List) value;
                if (valueList != null && valueList.size() > 0) {
                    value = valueList.get(0);
                }
            }
        }
        
        return (String) value;
    }
}
