/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.security;

import java.security.Principal;
import java.util.Iterator;
import java.util.Set;

import javax.security.auth.Subject;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.security.RunAsIdentity;
import org.jboss.security.SecurityAssociation;
import org.jboss.soa.esb.services.security.SecurityConfig;
import org.jboss.soa.esb.services.security.SecurityContext;
import org.jboss.soa.esb.services.security.principals.User;

/**
 * JBoss Application Server(AS) specifiec security context propagator.
 *
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public final class JBossASContextPropagator implements SecurityContextPropagator
{
    /**
     * Pushed the subject by calling {@link SecurityAssociation#pushSubjectContext(Subject, Principal, Object)}.
     *
     * @param context The SecurityContext holding the subject to push/propagate. May not be null.
     * @param config The SecurityConfig which contains the information specified from the security configuration element. Can be null.
     */
    public void pushSecurityContext(final SecurityContext context, Set<?> authCredentials, final SecurityConfig config)
    {
        AssertArgument.isNotNull(context, "context");

        final Subject subject = context.getSubject();
        final Principal principal = getPrincipal(subject);

        //  associate the subject with jboss security
        Object credential = null;
        if (authCredentials != null && authCredentials.isEmpty() == false)
        {
            credential = authCredentials.iterator().next();
        }
        SecurityAssociation.pushSubjectContext(subject, principal, credential);


        if (isRunAsSet(config))
        {
            //  associate the runAs role with jboss security
            SecurityAssociation.pushRunAsIdentity(new RunAsIdentity(config.getRunAs(), principal.getName()));
        }
    }

    /**
     * Pops the subject by calling {@link SecurityAssociation#pushSubjectContext(Subject, Principal, Object)}.
     *
     * @param context The SecurityContext holding the subject to push/propagate. Can be null.
     * @param config The SecurityConfig which contains the information specified from the security configuration element. Can be null.
     */
    public void popSecurityContext(final SecurityContext context, final SecurityConfig config)
    {
        //  diassociate the subject with jboss security
        SecurityAssociation.popSubjectContext();

        if (isRunAsSet(config))
        {
            //  associate the runAs role with jboss security
            SecurityAssociation.popRunAsIdentity();
        }
    }

    private Principal getPrincipal( final Subject subject)
    {
        for (Principal principal : subject.getPrincipals())
        {
            return principal;
        }
        return new User("NullPrincipal");
    }

    private boolean isRunAsSet(final SecurityConfig config)
    {
        return config != null && config.getRunAs() != null;
    }

}
