/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.util;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StringReader;
import java.util.Properties;

import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

/**
 * Utility functions for processing property files.
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
public class PropertiesHelper
{
    /**
     * Identity transform which adds the properties DOCTYPE to the output stream.
     */
    private static final String IDENTITY_TRANSFORM = 
        "<xsl:stylesheet xmlns:xsl='http://www.w3.org/1999/XSL/Transform' version='1.0'>" +
        "  <xsl:output method='xml' doctype-system='http://java.sun.com/dtd/properties.dtd'/>" +
        "  <xsl:template match='/'>" +
        "    <xsl:apply-templates/>" +
        "  </xsl:template>" +
        "  <xsl:template match='*|@*'>" +
        "    <xsl:copy>" +
        "      <xsl:apply-templates select='@*|node()'/>" +
        "    </xsl:copy>" +
        "  </xsl:template>" +
        "</xsl:stylesheet>" ;
        
    /**
     * Translate a property file from an XML format to a text format.
     * @param xmlPropertyFile The XML file
     * @param propertyFile The property file
     * @param propertyFileComments The property file comments
     * @return The properties
     * @throws IOException for errors during reading/writing.
     */
    public static Properties translateXMLToText(final File xmlPropertyFile,
        final File propertyFile, final String propertyFileComments)
        throws IOException
    {
        final FileInputStream fis = new FileInputStream(xmlPropertyFile) ;
        try
        {
            final FileOutputStream fos = new FileOutputStream(propertyFile) ;
            try
            {
                return translateXMLToText(fis, fos, propertyFileComments) ;
            }
            finally
            {
                
                fos.close() ;
            }
        }
        finally
        {
            fis.close() ;
        }
    }
    
    /**
     * Translate a property file from an XML format to a text format.
     * @param xmlPropertyInputStream The XML property input stream
     * @param propertyOutputStream The property output stream
     * @param propertyFileComments The property file comments
     * @return The properties
     * @throws IOException for errors during reading/writing.
     */
    public static Properties translateXMLToText(final InputStream xmlPropertyInputStream,
        final OutputStream propertyOutputStream, final String propertyFileComments)
        throws IOException
    {
        /*
         * Hack, to include the doctype into the property XML.
         * If this is included in the XML file then the ServiceBinding transformation
         * does not work when offline (cannot set URI resolvers).
         */
        final TransformerFactory transformerFactory = TransformerFactory.newInstance() ;
        final Source identitySource = new StreamSource(new StringReader(IDENTITY_TRANSFORM)) ;
        final Transformer transformer ;
        try
        {
            transformer = transformerFactory.newTransformer(identitySource) ;
        }
        catch (final TransformerConfigurationException tce)
        {
            final IOException ioe = new IOException("Failed to configure identity transformation") ;
            ioe.initCause(tce) ;
            throw ioe ;
        }
        
        final ByteArrayOutputStream result = new ByteArrayOutputStream() ;
        try
        {
            transformer.transform(new StreamSource(xmlPropertyInputStream), new StreamResult(result)) ;
        }
        catch (final TransformerException te)
        {
            final IOException ioe = new IOException("Failed to evaluate the identity transformation") ;
            ioe.initCause(te) ;
            throw ioe ;
        }
        // END of transformation hack
        
        final Properties xmlProperties = new Properties() ;
        final ByteArrayInputStream bais = new ByteArrayInputStream(result.toByteArray()) ;
        
        xmlProperties.loadFromXML(bais) ;
        
        final BufferedOutputStream bos = new BufferedOutputStream(propertyOutputStream) ;
        xmlProperties.store(bos, propertyFileComments) ;
        bos.flush() ;
        
        return xmlProperties ;
    }
}
